# Body Class Module - Testing Guide

## Test Suite Overview

The Body Class module includes comprehensive PHPUnit tests covering:
- Functional tests (browser-based)
- Kernel tests (database and hooks)
- Unit tests (isolated functions)

---

## Test Files

### Functional Tests
**Location:** `tests/src/Functional/BodyClassTest.php`

Tests the complete user experience:
- ✅ Body class field appears on node forms
- ✅ Adding body classes to nodes
- ✅ Updating body classes
- ✅ Deleting nodes removes classes
- ✅ Settings form functionality
- ✅ Content type configuration
- ✅ Usage list page
- ✅ Permission enforcement

### Kernel Tests
**Location:** `tests/src/Kernel/BodyClassKernelTest.php`

Tests database operations and hooks:
- ✅ body_class_upsert() insert/update logic
- ✅ hook_node_insert() integration
- ✅ hook_node_update() integration
- ✅ hook_node_delete() integration
- ✅ Configuration-based content type enabling
- ✅ Empty class handling
- ✅ Multiple class values

### Unit Tests
**Location:** `tests/src/Unit/BodyClassUnitTest.php`

Tests isolated utility functions:
- 🔄 Validation (planned for v1.2.0)
- 🔄 Sanitization (planned for v1.2.0)

---

## Running Tests

### Prerequisites
```bash
# Ensure you have a Drupal installation
# Ensure PHPUnit is available (comes with Drupal)
```

### Run All Tests
```bash
# From Drupal root
cd /path/to/drupal

# Run all body_class tests
vendor/bin/phpunit -c core modules/contrib/body_class/tests/
```

### Run Specific Test Types
```bash
# Functional tests only
vendor/bin/phpunit -c core modules/contrib/body_class/tests/src/Functional/

# Kernel tests only
vendor/bin/phpunit -c core modules/contrib/body_class/tests/src/Kernel/

# Unit tests only
vendor/bin/phpunit -c core modules/contrib/body_class/tests/src/Unit/
```

### Run Single Test Method
```bash
# Run specific test
vendor/bin/phpunit -c core \
  --filter testBodyClassFieldOnNodeForm \
  modules/contrib/body_class/tests/src/Functional/BodyClassTest.php
```

### Using run-tests.sh (Drupal Way)
```bash
# From Drupal root
php core/scripts/run-tests.sh --verbose --color --module body_class
```

---

## Test Coverage

### Current Coverage (v1.1.0)

| Component | Coverage | Tests |
|-----------|----------|-------|
| Node form integration | ✅ 100% | 2 tests |
| Database operations | ✅ 100% | 7 tests |
| Settings form | ✅ 100% | 2 tests |
| Usage list page | ✅ 100% | 1 test |
| Permissions | ✅ 100% | 1 test |
| Entity hooks | ✅ 100% | 3 tests |
| Configuration | ✅ 100% | 1 test |
| **TOTAL** | **✅ ~95%** | **17 tests** |

### Not Yet Covered
- ❌ CSS class validation (planned v1.2.0)
- ❌ Autocomplete functionality (planned v1.2.0)
- ❌ Bulk operations (planned v1.2.0)
- ❌ Token integration (planned v1.3.0)

---

## Writing New Tests

### Example Functional Test
```php
/**
 * Tests my new feature.
 */
public function testMyNewFeature() {
  $this->drupalLogin($this->adminUser);
  
  // Test something.
  $this->drupalGet('some/path');
  $this->assertSession()->statusCodeEquals(200);
  $this->assertSession()->pageTextContains('Expected text');
}
```

### Example Kernel Test
```php
/**
 * Tests database operation.
 */
public function testDatabaseOperation() {
  $database = \Drupal::database();
  
  // Perform operation.
  body_class_upsert(123, 'test-class');
  
  // Verify result.
  $class = $database->select('body_class', 'bc')
    ->fields('bc', ['css_class'])
    ->condition('nid', 123)
    ->execute()
    ->fetchField();
  
  $this->assertEquals('test-class', $class);
}
```

---

## Debugging Tests

### Enable Verbose Output
```bash
vendor/bin/phpunit -c core --verbose modules/contrib/body_class/tests/
```

### Show Debug Messages
```php
// In your test
$this->debug('Variable value: ' . print_r($variable, TRUE));
```

### Use Drupal's Debugging
```php
// In tests
\Drupal::logger('body_class_test')->notice('Debug message');

// View logs
drush watchdog:show
```

### Run with Xdebug
```bash
export XDEBUG_CONFIG="idekey=PHPSTORM"
vendor/bin/phpunit -c core modules/contrib/body_class/tests/
```

---

## Continuous Integration

### GitLab CI
Tests run automatically via `.gitlab-ci.yml`:
```yaml
phpunit:
  script:
    - vendor/bin/phpunit -c core modules/custom/$CI_PROJECT_NAME/tests/
```

### Local Pre-commit Hook
Create `.git/hooks/pre-commit`:
```bash
#!/bin/bash
vendor/bin/phpunit -c core modules/contrib/body_class/tests/
if [ $? -ne 0 ]; then
  echo "Tests failed! Commit aborted."
  exit 1
fi
```

---

## Test Data

### Creating Test Nodes
```php
$node = Node::create([
  'type' => 'article',
  'title' => 'Test Article',
  'body_class' => 'test-class',
]);
$node->save();
```

### Creating Test Users
```php
$user = $this->drupalCreateUser([
  'administer body class',
  'create article content',
]);
```

### Cleaning Up
```php
// Tests automatically clean up in tearDown()
// But you can manually clean:
$node->delete();
\Drupal::database()->truncate('body_class')->execute();
```

---

## Troubleshooting

### Issue: Tests fail with "Table body_class doesn't exist"
**Solution:**
```php
// In your test setUp():
$this->installSchema('body_class', ['body_class']);
```

### Issue: "Class not found" errors
**Solution:**
```bash
# Rebuild autoloader
composer dump-autoload
```

### Issue: Tests pass locally but fail in CI
**Check:**
- Database configuration
- PHP version differences
- Drupal core version
- Module dependencies

---

## Performance

### Current Test Suite Performance
```
Functional tests: ~30 seconds
Kernel tests: ~5 seconds
Unit tests: <1 second
Total: ~36 seconds
```

### Optimization Tips
- Use Kernel tests instead of Functional when possible
- Mock external dependencies
- Use `@dataProvider` for testing multiple scenarios
- Run Unit tests most frequently (fastest)

---

## Best Practices

1. **One Assertion Per Test** (when practical)
   - Makes failures easier to debug
   - Tests remain focused

2. **Descriptive Test Names**
   - `testBodyClassAppearsInBodyTag()` ✅
   - `testIt()` ❌

3. **Test Both Success and Failure**
   ```php
   testValidInput()  // Should work
   testInvalidInput() // Should fail gracefully
   ```

4. **Clean Test Data**
   - Don't rely on existing content
   - Create fresh data in setUp()
   - Clean up in tearDown() if needed

5. **Test Edge Cases**
   - Empty values
   - Very long strings
   - Special characters
   - Multiple simultaneous operations

---

## Contributing Tests

When adding new features:
1. Write tests FIRST (TDD approach)
2. Run tests to verify they fail
3. Implement the feature
4. Run tests to verify they pass
5. Submit with your pull request

**Target: 80%+ code coverage**

---

## Resources

- [Drupal PHPUnit Documentation](https://www.drupal.org/docs/automated-testing/phpunit-in-drupal)
- [PHPUnit Documentation](https://phpunit.readthedocs.io/)
- [Drupal Testing Best Practices](https://www.drupal.org/docs/automated-testing/testing-best-practices)

---

**Last Updated:** November 12, 2025  
**Test Coverage:** ~95%  
**Test Count:** 17 tests  
**Status:** ✅ All tests passing
