<?php

namespace Drupal\Tests\body_class\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests the Body Class module functionality.
 *
 * @group body_class
 */
// phpcs:ignore Drupal.Commenting.Deprecated.DeprecatedMissing
#[\PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses]
class BodyClassTest extends BrowserTestBase
{

    /**
     * {@inheritdoc}
     */
    protected $defaultTheme = 'stark';

    /**
     * {@inheritdoc}
     */
    protected static $modules = ['node', 'body_class'];

    /**
     * A user with permission to administer body class.
     *
     * @var \Drupal\user\UserInterface
     */
    protected $adminUser;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        // Create Article content type.
        NodeType::create(
            [
            'type' => 'article',
            'name' => 'Article',
            ]
        )->save();

        // Create Page content type.
        NodeType::create(
            [
            'type' => 'page',
            'name' => 'Page',
            ]
        )->save();

        // Create a user with admin permissions.
        $this->adminUser = $this->drupalCreateUser(
            [
            'administer body class',
            'administer nodes',
            'create article content',
            'edit any article content',
            'create page content',
            'edit any page content',
            ]
        );
    }

    /**
     * Tests that the body class field appears on node forms.
     */
    public function testBodyClassFieldOnNodeForm()
    {
        $this->drupalLogin($this->adminUser);

        // Visit article creation form.
        $this->drupalGet('node/add/article');
        $this->assertSession()->statusCodeEquals(200);
        $this->assertSession()->fieldExists('body_class');

        // Visit page creation form.
        $this->drupalGet('node/add/page');
        $this->assertSession()->statusCodeEquals(200);
        $this->assertSession()->fieldExists('body_class');
    }

    /**
     * Tests adding a body class to a node.
     */
    public function testAddBodyClassToNode()
    {
        $this->drupalLogin($this->adminUser);

        // Create a node with a body class.
        $edit = [
        'title[0][value]' => 'Test Article',
        'body_class' => 'test-class featured',
        ];
        $this->drupalGet('node/add/article');
        $this->submitForm($edit, 'Save');

        // Verify the node was created.
        $this->assertSession()->pageTextContains('Test Article');

        // Load the node and verify the class is stored in the database.
        $nodes = \Drupal::entityTypeManager()
            ->getStorage('node')
            ->loadByProperties(['title' => 'Test Article']);
        $node = reset($nodes);

        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();

        $this->assertEquals('test-class featured', $class);

        // Visit the node page and check if the class appears in the body tag.
        $this->drupalGet('node/' . $node->id());
        $this->assertSession()->responseContains('test-class');
        $this->assertSession()->responseContains('featured');
    }

    /**
     * Tests updating a body class on an existing node.
     */
    public function testUpdateBodyClass()
    {
        // Create a node programmatically.
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();

        // Add body class via helper function.
        body_class_upsert($node->id(), 'original-class');

        // Verify the original class is stored.
        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();
        $this->assertEquals('original-class', $class);

        // Update the node with a new class.
        $this->drupalLogin($this->adminUser);
        $this->drupalGet('node/' . $node->id() . '/edit');
        $this->submitForm(['body_class' => 'updated-class'], 'Save');

        // Verify the class was updated.
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();
        $this->assertEquals('updated-class', $class);
    }

    /**
     * Tests deleting a node removes the body class from database.
     */
    public function testDeleteNodeRemovesBodyClass()
    {
        // Create a node with a body class.
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();
        $nid = $node->id();

        // Add body class via helper function.
        body_class_upsert($nid, 'delete-test');

        // Verify the class exists in the database.
        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $nid)
            ->execute()
            ->fetchField();
        $this->assertEquals('delete-test', $class);

        // Delete the node.
        $node->delete();

        // Verify the class was removed from the database.
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $nid)
            ->execute()
            ->fetchField();
        $this->assertFalse($class);
    }

    /**
     * Tests the settings configuration form.
     */
    public function testSettingsForm()
    {
        $this->drupalLogin($this->adminUser);

        // Visit the settings page.
        $this->drupalGet('admin/config/development/body_class');
        $this->assertSession()->statusCodeEquals(200);
        $this->assertSession()->pageTextContains('Body Class Settings');

        // Verify the enable_all checkbox exists and is checked by default.
        $this->assertSession()->checkboxChecked('enable_all');

        // Verify content type checkboxes exist.
        $this->assertSession()->fieldExists('enabled_content_types[article]');
        $this->assertSession()->fieldExists('enabled_content_types[page]');
    }

    /**
     * Tests disabling body class for specific content types.
     */
    public function testContentTypeConfiguration()
    {
        $this->drupalLogin($this->adminUser);

        // Configure to enable only for articles.
        $this->drupalGet('admin/config/development/body_class');
        $edit = [
        'enable_all' => false,
        'enabled_content_types[article]' => 'article',
        'enabled_content_types[page]' => false,
        ];
        $this->submitForm($edit, 'Save configuration');
        $this->assertSession()->pageTextContains('The configuration options have been saved.');

        // Clear cache to rebuild field definitions.
        drupal_flush_all_caches();

        // Visit article form - field should exist.
        $this->drupalGet('node/add/article');
        $this->assertSession()->fieldExists('body_class');

        // Visit page form - field should not exist.
        $this->drupalGet('node/add/page');
        $this->assertSession()->fieldNotExists('body_class');
    }

    /**
     * Tests the usage list page.
     */
    public function testUsageListPage()
    {
        // Create some nodes with body classes.
        $node1 = Node::create(
            [
            'type' => 'article',
            'title' => 'Node 1',
            ]
        );
        $node1->save();
        body_class_upsert($node1->id(), 'class-one');

        $node2 = Node::create(
            [
            'type' => 'article',
            'title' => 'Node 2',
            ]
        );
        $node2->save();
        body_class_upsert($node2->id(), 'class-two');

        $node3 = Node::create(
            [
            'type' => 'article',
            'title' => 'Node 3',
            ]
        );
        $node3->save();

        $this->drupalLogin($this->adminUser);

        // Visit the usage list page.
        $this->drupalGet('admin/config/development/body_class/list');
        $this->assertSession()->statusCodeEquals(200);

        // Verify nodes with classes are shown.
        $this->assertSession()->pageTextContains('Node 1');
        $this->assertSession()->pageTextContains('class-one');
        $this->assertSession()->pageTextContains('Node 2');
        $this->assertSession()->pageTextContains('class-two');

        // Verify node without class is not shown.
        $this->assertSession()->pageTextNotContains('Node 3');
    }

    /**
     * Tests permission enforcement.
     */
    public function testPermissions()
    {
        // Create a user without body class admin permission.
        $normalUser = $this->drupalCreateUser(
            [
            'create article content',
            'edit own article content',
            ]
        );

        $this->drupalLogin($normalUser);

        // User should not be able to access settings page.
        $this->drupalGet('admin/config/development/body_class');
        $this->assertSession()->statusCodeEquals(403);

        // User should not be able to access usage list page.
        $this->drupalGet('admin/config/development/body_class/list');
        $this->assertSession()->statusCodeEquals(403);

        // User should not see the body_class field on node forms.
        $this->drupalGet('node/add/article');
        $this->assertSession()->fieldNotExists('body_class');
    }

}