<?php

namespace Drupal\Tests\body_class\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests Body Class database operations and hooks.
 *
 * @group body_class
 */
// phpcs:ignore Drupal.Commenting.Deprecated.DeprecatedMissing
#[\PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses]
class BodyClassKernelTest extends KernelTestBase
{

    /**
     * {@inheritdoc}
     */
    protected static $modules = ['system', 'user', 'node', 'body_class', 'field', 'text'];

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->installEntitySchema('user');
        $this->installEntitySchema('node');
        $this->installSchema('node', ['node_access']);
        $this->installSchema('body_class', ['body_class']);
        $this->installConfig(['node', 'body_class']);

        // Create article content type.
        NodeType::create(
            [
            'type' => 'article',
            'name' => 'Article',
            ]
        )->save();
    }

    /**
     * Tests the body_class_upsert() function.
     */
    public function testBodyClassUpsert()
    {
        $database = \Drupal::database();

        // Test insert operation.
        body_class_upsert(1, 'test-class');

        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', 1)
            ->execute()
            ->fetchField();

        $this->assertEquals('test-class', $class);

        // Test update operation.
        body_class_upsert(1, 'updated-class');

        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', 1)
            ->execute()
            ->fetchField();

        $this->assertEquals('updated-class', $class);

        // Test that only one record exists.
        $count = $database->select('body_class', 'bc')
            ->condition('nid', 1)
            ->countQuery()
            ->execute()
            ->fetchField();

        $this->assertEquals(1, $count);
    }

    /**
     * Tests hook_node_insert() integration.
     */
    public function testNodeInsertHook()
    {
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();

        // Manually insert body class (simulating what form submit would do).
        body_class_upsert($node->id(), 'insert-test');

        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();

        $this->assertEquals('insert-test', $class);
    }

    /**
     * Tests hook_node_update() integration.
     */
    public function testNodeUpdateHook()
    {
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();

        // Insert initial body class.
        body_class_upsert($node->id(), 'original');

        // Update the body class.
        body_class_upsert($node->id(), 'updated');

        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();

        $this->assertEquals('updated', $class);
    }

    /**
     * Tests hook_node_delete() integration.
     */
    public function testNodeDeleteHook()
    {
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();
        $nid = $node->id();

        // Insert body class.
        body_class_upsert($nid, 'delete-me');

        // Verify class exists.
        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $nid)
            ->execute()
            ->fetchField();
        $this->assertEquals('delete-me', $class);

        // Delete the node.
        $node->delete();

        // Verify class was removed.
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $nid)
            ->execute()
            ->fetchField();
        $this->assertFalse($class);
    }

    /**
     * Tests configuration-based content type enabling.
     */
    public function testContentTypeConfiguration()
    {
        // Create page content type.
        NodeType::create(
            [
            'type' => 'page',
            'name' => 'Page',
            ]
        )->save();

        // Set configuration to enable only articles.
        \Drupal::configFactory()->getEditable('body_class.settings')
            ->set('enabled_content_types', ['article'])
            ->save();

        // Clear field definition cache.
        \Drupal::service('entity_field.manager')->clearCachedFieldDefinitions();

        // Create article node - should save class.
        $article = Node::create(
            [
            'type' => 'article',
            'title' => 'Article Test',
            ]
        );
        $article->save();

        // Insert body class for article.
        body_class_upsert($article->id(), 'article-class');

        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $article->id())
            ->execute()
            ->fetchField();

        $this->assertEquals('article-class', $class);

        // Create page node - body_class field shouldn't be available.
        $page = Node::create(
            [
            'type' => 'page',
            'title' => 'Page Test',
            ]
        );
        $page->save();

        // Page should not have an entry in body_class table.
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $page->id())
            ->execute()
            ->fetchField();

        $this->assertFalse($class);
    }

    /**
     * Tests empty class handling.
     */
    public function testEmptyClassHandling()
    {
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();

        // Insert initial class.
        body_class_upsert($node->id(), 'initial-class');

        // Update to empty class.
        body_class_upsert($node->id(), '');

        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();

        // Empty string should be stored.
        $this->assertEquals('', $class);
    }

    /**
     * Tests multiple class values.
     */
    public function testMultipleClasses()
    {
        $node = Node::create(
            [
            'type' => 'article',
            'title' => 'Test Node',
            ]
        );
        $node->save();

        // Insert multiple classes.
        body_class_upsert($node->id(), 'class-one class-two class-three');

        $database = \Drupal::database();
        $class = $database->select('body_class', 'bc')
            ->fields('bc', ['css_class'])
            ->condition('nid', $node->id())
            ->execute()
            ->fetchField();

        $this->assertEquals('class-one class-two class-three', $class);
    }

}
