<?php

/**
 * @file
 * Post update functions for the book module.
 */

use Drupal\Core\Config\Entity\ConfigEntityUpdater;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\block\Entity\Block;
use Drupal\user\RoleInterface;

/**
 * Pre-populate the use_top_level_title setting of the book_navigation blocks.
 */
function book_post_update_prepopulate_use_top_level_title_setting(&$sandbox): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'block', function (Block $block) {
    if ($block->getPluginId() === 'book_navigation') {
      $block->getPlugin()->setConfigurationValue('use_top_level_title', FALSE);
      return TRUE;
    }
    return FALSE;
  });
}

/**
 * Update extra book field for entity view displays.
 */
function book_post_update_book_navigation_view_display(?array &$sandbox = NULL): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'entity_view_display', function (EntityViewDisplayInterface $entity_view_display): bool {
    $update = FALSE;
    $components = $entity_view_display->getComponents();
    if ($entity_view_display->getTargetEntityTypeId() === 'node') {
      if (isset($components['book_navigation'])) {
        if ($entity_view_display->getMode() !== 'full' || $entity_view_display->getMode() !== 'default') {
          $updated_component = $entity_view_display->getComponent('book_navigation');
          $updated_component['region'] = 'hidden';
          $entity_view_display->setComponent('book_navigation', $updated_component);
          $update = TRUE;
        }
      }
    }
    return $update;
  });
}

/**
 * Grant 'access book list' permission to roles with 'access content'.
 */
function book_post_update_book_list_permission(?array &$sandbox = NULL): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'user_role', function (RoleInterface $role): bool {
    $update = FALSE;

    if ($role->hasPermission('access content') && !$role->hasPermission('access book list')) {
      $role->grantPermission('access book list');
      $update = TRUE;
    }

    return $update;
  });
}

/**
 * Converts book.settings.allowed_types + child_type to structured format.
 */
function book_post_update_convert_allowed_types_to_structured(array &$sandbox = []): void {
  $config_factory = \Drupal::configFactory();
  $config = $config_factory->getEditable('book.settings');

  $old_allowed = $config->get('allowed_types');
  $child_type = $config->get('child_type');

  // Only run if we're still using the old flat format.
  if (is_array($old_allowed) && isset($child_type)) {
    $new_allowed = [];
    foreach ($old_allowed as $type_id) {
      if (!empty($type_id)) {
        $new_allowed[] = [
          'content_type' => $type_id,
          'child_type' => $child_type,
        ];
      }
    }

    $config
      ->set('allowed_types', $new_allowed)
      ->clear('child_type')
      ->save();
  }
}

/**
 * Pre-populate the show_top_item setting of the book_navigation blocks.
 */
function book_post_update_prepopulate_show_top_item_setting(&$sandbox): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'block', function (Block $block) {
    if ($block->getPluginId() === 'book_navigation') {
      $block->getPlugin()->setConfigurationValue('show_top_item', FALSE);
      return TRUE;
    }
    return FALSE;
  });
}

/**
 * Add the default label truncation setting.
 */
function book_post_update_add_default_label_truncate_settings(): ?string {
  $config = \Drupal::configFactory()->getEditable('book.settings');
  if ($config->get('truncate_label') === NULL) {
    // Specify the default setting if not specified.
    $config
      ->set('truncate_label', TRUE)
      ->save(TRUE);
    return 'Updated the default truncate setting.';
  }
  return NULL;
}
