<?php
declare(strict_types=1);

namespace Drupal\book_organizer\Plugin\views\style;

use Drupal\book_organizer\Service\BookHierarchyManager;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a hierarchical table renderer for book nodes.
 *
 * @ViewsStyle(
 *   id = "book_hierarchy_table",
 *   title = @Translation("Book hierarchy table"),
 *   help = @Translation("Displays books with their hierarchical pages in a single table."),
 *   display_types = {"normal"}
 * )
 */
final class BookHierarchyTable extends StylePluginBase implements ContainerFactoryPluginInterface {

  /**
   * Hierarchy helper service.
   */
  protected BookHierarchyManager $hierarchyManager;

  /**
   * Date formatter.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('book_organizer.hierarchy_manager'),
      $container->get('date.formatter'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, BookHierarchyManager $hierarchy_manager, DateFormatterInterface $date_formatter) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->hierarchyManager = $hierarchy_manager;
    $this->dateFormatter    = $date_formatter;
    $this->usesRowPlugin    = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function render(): array {
    $build = [
      '#theme'      => 'table',
      '#header'     => [
        $this->t('Title'),
        $this->t('Author'),
        $this->t('Updated'),
        $this->t('Operations'),
      ],
      '#rows'       => [],
      '#attributes' => ['class' => ['book-organized-table']],
      '#empty'      => $this->t('No books available.'),
    ];

    $books               = [];
    $topLevelIdsToLoad   = [];

    foreach ($this->view->result as $index => $row) {
      $node = $row->_entity ?? NULL;
      if (!$node instanceof NodeInterface || !$node->access('view')) {
        continue;
      }

      $inHierarchy = $this->hierarchyManager->isInBookHierarchy($node);
      $topLevelId  = $this->hierarchyManager->getTopLevelId($node);
      $isTopLevel  = $inHierarchy && (int) $node->id() === $topLevelId;

      if (!isset($books[$topLevelId])) {
        $books[$topLevelId] = [
          'top'      => NULL,
          'children' => [],
          'position' => $index,
        ];
      }
      else {
        $books[$topLevelId]['position'] = min($books[$topLevelId]['position'], $index);
      }

      if ($isTopLevel || !$inHierarchy) {
        $books[$topLevelId]['top'] = $node;
      }
      else {
        $books[$topLevelId]['children'][(int) $node->id()] = TRUE;
      }

      if ($books[$topLevelId]['top'] === NULL) {
        $topLevelIdsToLoad[$topLevelId] = $topLevelId;
      }
    }

    if ($topLevelIdsToLoad !== []) {
      $loaded = $this->hierarchyManager->loadMultiple($topLevelIdsToLoad);
      foreach ($topLevelIdsToLoad as $id) {
        if (!isset($books[$id]['top']) && isset($loaded[$id]) && $loaded[$id] instanceof NodeInterface) {
          $books[$id]['top'] = $loaded[$id];
        }
      }
    }

    foreach ($books as $id => $data) {
      if (!$data['top'] instanceof NodeInterface || !$data['top']->access('view')) {
        unset($books[$id]);
      }
    }

    uasort($books, static function (array $a, array $b): int {
      return $a['position'] <=> $b['position'];
    });

    foreach ($books as $data) {
      /** @var \Drupal\node\NodeInterface $topNode */
      $topNode         = $data['top'];
      $build['#rows'][] = $this->buildTopLevelRow($topNode);

      if (!$this->hierarchyManager->isTopLevelBook($topNode)) {
        continue;
      }

      $pages = $this->hierarchyManager->getBookPages($topNode->id());
      if ($pages === []) {
        continue;
      }

      if ($data['children'] === []) {
        continue;
      }

      foreach ($pages as $page) {
        if ($page->id() === $topNode->id()) {
          continue;
        }
        if (!isset($data['children'][(int) $page->id()])) {
          continue;
        }
        if (!$page instanceof NodeInterface || !$page->access('view')) {
          continue;
        }

        $build['#rows'][] = $this->buildChildRow($page);
      }
    }

    return $build;
  }

  /**
   * Builds the render array for a top-level book row.
   */
  protected function buildTopLevelRow(NodeInterface $node): array {
    $isInHierarchy = $this->hierarchyManager->isTopLevelBook($node);
    $titleSuffix   = '';
    $allowOrder    = FALSE;

    if ($isInHierarchy) {
      $pageCount   = $this->hierarchyManager->getBookPageCount($node->id());
      $titleSuffix = $this->formatPlural($pageCount, '(@count page)', '(@count pages)');
      $allowOrder  = $pageCount > 1;
    }
    else {
      $titleSuffix = '<em>(' . $this->t('standalone') . ')</em>';
    }

    $titleMarkup = '<strong>' . $node->toLink()->toString();
    if ($titleSuffix !== '') {
      $titleMarkup .= ' ' . $titleSuffix;
    }
    $titleMarkup .= '</strong>';

    $operations = $this->buildOperations($node, $allowOrder);

    return [
      'data'  => [
        ['data' => ['#markup' => $titleMarkup]],
        [
          'data' => $node->getOwner()
            ? $node->getOwner()->toLink()
            : $this->t('Anonymous'),
        ],
        [
          'data' => [
            '#markup' => $this->dateFormatter->format($node->getChangedTime(), 'short'),
          ],
        ],
        [
          'data' => [
            '#type'  => 'operations',
            '#links' => $operations,
          ],
        ],
      ],
      'class' => ['book-heading'],
    ];
  }

  /**
   * Builds the render array for a child book page row.
   */
  protected function buildChildRow(NodeInterface $node): array {
    $depth  = !empty($node->book['depth']) ? (int) $node->book['depth'] : 0;
    $indent = str_repeat('—', max($depth - 1, 0));

    $operations = $this->buildOperations($node, !empty($node->book['has_children']));

    return [
      'data' => [
        [
          'data' => [
            '#markup' => trim($indent . ' ' . $node->toLink()->toString()),
          ],
        ],
        [
          'data' => $node->getOwner()
            ? $node->getOwner()->toLink()
            : $this->t('Anonymous'),
        ],
        [
          'data' => [
            '#markup' => $this->dateFormatter->format($node->getChangedTime(), 'short'),
          ],
        ],
        [
          'data' => [
            '#type'  => 'operations',
            '#links' => $operations,
          ],
        ],
      ],
    ];
  }

  /**
   * Builds operations for a node, optionally adding the order link.
   */
  protected function buildOperations(NodeInterface $node, bool $include_order): array {
    $operations = [];

    if ($node->access('update')) {
      $operations['edit'] = [
        'title' => $this->t('Edit'),
        'url'   => $node->toUrl('edit-form'),
      ];
    }

    if ($node->access('view')) {
      $operations['view'] = [
        'title' => $this->t('View'),
        'url'   => $node->toUrl(),
      ];
    }

    if ($include_order && $node->access('update')) {
      $operations['order'] = [
        'title' => $this->t('Order pages'),
        'url'   => Url::fromRoute('book.admin_edit', ['node' => $node->id()]),
      ];
    }

    return $operations;
  }

}
