<?php
declare(strict_types=1);

namespace Drupal\book_organizer\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;

/**
 * Provides helpers for working with book hierarchies.
 */
final class BookHierarchyManager {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * Constructs the hierarchy manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, Connection $database) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database          = $database;
  }

  /**
   * Gets the number of pages in a book.
   */
  public function getBookPageCount(int|string $bid): int {
    $bid = (int) $bid;

    return (int) $this->database
      ->select('book', 'b')
      ->condition('b.bid', $bid)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Loads all pages that belong to the given book in hierarchical order.
   *
   * @return \Drupal\node\NodeInterface[]
   *   The ordered list of book pages.
   */
  public function getBookPages(int|string $bid): array {
    $bid = (int) $bid;

    $query = $this->database
      ->select('book', 'b')
      ->fields('b', ['nid'])
      ->condition('b.bid', $bid)
      ->orderBy('b.weight', 'ASC')
      ->orderBy('b.p1', 'ASC')
      ->orderBy('b.p2', 'ASC')
      ->orderBy('b.p3', 'ASC')
      ->orderBy('b.p4', 'ASC')
      ->orderBy('b.p5', 'ASC')
      ->orderBy('b.p6', 'ASC')
      ->orderBy('b.p7', 'ASC')
      ->orderBy('b.p8', 'ASC')
      ->orderBy('b.p9', 'ASC');

    $nids = $query->execute()->fetchCol();

    if ($nids === []) {
      return [];
    }

    $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($nids);

    $ordered = [];
    foreach ($nids as $nid) {
      if (isset($nodes[$nid])) {
        $ordered[$nid] = $nodes[$nid];
      }
    }

    return $ordered;
  }

  /**
   * Determines whether the provided node is part of a book hierarchy.
   */
  public function isInBookHierarchy(NodeInterface $node): bool {
    return isset($node->book) && !empty($node->book['bid']);
  }

  /**
   * Gets the top-level book ID for the provided node.
   */
  public function getTopLevelId(NodeInterface $node): int {
    if (!$this->isInBookHierarchy($node)) {
      return (int) $node->id();
    }
    return (int) $node->book['bid'];
  }

  /**
   * Checks whether the node represents the top-level entry of its book.
   */
  public function isTopLevelBook(NodeInterface $node): bool {
    if (!$this->isInBookHierarchy($node)) {
      return FALSE;
    }

    return (int) $node->id() === (int) $node->book['bid'];
  }

  /**
   * Loads multiple nodes keyed by their IDs.
   *
   * @param int[] $nids
   *   The node IDs to load.
   *
   * @return \Drupal\node\NodeInterface[]
   *   Loaded nodes keyed by ID.
   */
  public function loadMultiple(array $nids): array {
    $nids = array_values(array_unique(array_filter($nids, static fn ($value) => $value !== NULL && $value !== '')));
    if ($nids === []) {
      return [];
    }

    return $this->entityTypeManager->getStorage('node')->loadMultiple($nids);
  }

}
