# Bootstrap Components (Lightweight SDC Library)

A lightweight collection of **Bootstrap 5 Single Directory Components (SDC)** for Drupal 10/11.

> Most component structure and ideas are adapted / inspired by the `ui_suite_bootstrap` project. All primary credit for the original patterns belongs to that team – this module intentionally provides a slimmer, easier‑to‑audit subset without the broader ecosystem overhead.

## Features

- Modern **SDC** definitions (modal, alert, button, close-button starter set)
- Clean **`to_attributes` Twig filter/function** for reliable attribute normalization
- Optional variant & slot patterns consistent with Drupal core SDC direction
- Zero JS build step required (relies on Bootstrap already present on your site)

## Twig Helper: `to_attributes`

Added via `AttributesToolTwigExtension` and exposed as both a filter & function.

Usage examples:
```twig
{# From a space-delimited string #}
<div{{ 'alpha beta'|to_attributes }}></div>

{# From an array of classes #}
<div{{ ['alpha','beta']|to_attributes }}></div>

{# From an associative array #}
<div{{ { class: ['alpha','beta'], id: 'example', 'data-role': 'dialog' }|to_attributes }}></div>
```

It always returns a `Drupal\Core\Template\Attribute` object, making it safe to chain `.addClass()` in Twig.

## Installation

1. Place this module in: `docroot/modules/custom/bootstrap_components`
2. Ensure Bootstrap (CSS & JS + Popper) is loaded globally (theme or library).
3. Enable the module:
```bash
drush en bootstrap_components -y
```
4. Clear caches:
```bash
drush cr
```

## Using Components in Twig

Use Drupal's Single Directory Component (SDC) Twig patterns. Prefer `include` for simple one-off components without slot content, and `embed` when you need to populate slot (block) content.

Official docs: https://www.drupal.org/docs/develop/theming-drupal/using-single-directory-components/using-your-new-single-directory-component

### Simple include (button)
If a component only needs a label (slot) you can either embed and fill the block or—when the component is written to accept a prop—pass it directly. This button defines a `label` slot, so we use `embed` to supply it.

```twig
{% embed 'bootstrap_components:button' with { variant: 'primary' } only %}
  {% block label %}Click Me{% endblock %}
{% endembed %}
```

If you create a component that uses simple props only (no slot blocks) you can include it directly:
```twig
{{ include('bootstrap_components:close_button', { label: 'Dismiss', attributes: { class: ['float-end'] } }, with_context = false) }}
```

### Component with multiple slot regions (modal)
The modal exposes `title`, `body`, and `footer` slots. Provide them via `embed` blocks; pass any props (e.g. centered, scrollable, variant) in the `with` hash.

```twig
{% embed 'bootstrap_components:modal' with { centered: true, scrollable: true, variant: 'lg' } only %}
  {% block title %}Example Modal{% endblock %}
  {% block body %}
    <p>This is the body content of the modal. You can place arbitrary markup here.</p>
    {% embed 'bootstrap_components:button' with { variant: 'secondary' } only %}
      {% block label %}Secondary Action{% endblock %}
    {% endembed %}
  {% endblock %}
  {% block footer %}
    {% embed 'bootstrap_components:button' with { variant: 'primary' } only %}
      {% block label %}Save changes{% endblock %}
    {% endembed %}
    {% embed 'bootstrap_components:button' with { variant: 'secondary' } only %}
      {% block label %}Close{% endblock %}
    {% endembed %}
  {% endblock %}
{% endembed %}
```

### Nested components (accordion inside card, illustrative)
```twig
{% embed 'bootstrap_components:accordion' with { always_open: true } only %}
  {% block default %}
    {# Provide accordion items via an include or further embeds depending on your implementation. #}
  {% endblock %}
{% endembed %}
```

Guidelines:
- Always use `only` to avoid leaking parent context unexpectedly.
- Use `with_context = false` on `include()` when you want isolation.
- Keep variant logic (e.g. `variant: 'primary'`) explicit rather than implied by surrounding markup.

## Stories

Story YAML files (e.g. `modal.default.story.yml`) live under each component `stories/` folder and are discoverable by a consumer (for example an in-project UI Kit or other rendering tool). They follow this filename pattern:
```
{component}.{key}.story.yml
```

Example excerpt:
```yaml
name: "Default modal"
props:
  title: "Example"
  body: "<p>Lorem ipsum</p>"
  animation: true
```

## Stories & `library_wrapper`

Component authors can visually wrap rendered stories in a consumer (UI Kit or other) by adding a `library_wrapper:` key to the story YAML. The wrapper may be:

- An inline Twig fragment (must use the available context vars or `{{ _story }}` to render the component):
```yaml
library_wrapper: "<div class='uikit-preview p-3 bg-light'>{{ _story }}</div>"
```

- An @-include reference to a Twig template (recommended for complex wrappers). Example include namespace shown below is illustrative — point at whatever namespace your consumer exposes, e.g. a UI Kit or theme that provides a wrapper template:
```yaml
library_wrapper: "@bootstrap_ui_kit/ui-kit/modal-wrapper.twig"
```

Wrapper context variables available to the fragment/template:
- `_story` — the render array (or markup) of the component
- `_props` — final merged props for the story (story defaults + user overrides)
- `_slots` — transformed slot render arrays
- `_component_id` — full provider:machine component id

Notes:
- Always include a top-level `name:` in your story YAML so discovery will list it.
- If you save a full filesystem path as the component's `_story` prop in a consumer configuration, that path is used directly (it must be readable by PHP).

## Credits

- Heavy inspiration & numerous original patterns: **ui_suite_bootstrap** (thank you!).
- Drupal SDC / Component initiative.

## License

MIT (see component source headers where applicable). If you adapt more code directly from `ui_suite_bootstrap`, please retain their attribution per file.

## Roadmap / Ideas

- Additional core Bootstrap components (accordion, offcanvas, navbar, toast)
- Optional accessibility audit helper output in dev mode
- Expanded story metadata for automated visual regression harnesses

Pull requests / issues welcome.
