<?php

namespace Drupal\bootstrap_components\Twig;

use Drupal\Core\Template\Attribute;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;

/**
 * Provides a |to_attributes filter & function to normalize attribute input.
 *
 * Accepts:
 *  - Attribute object (returned as-is)
 *  - Associative array (converted to Attribute)
 *  - Sequential array of classes (converted to Attribute(['class'=>[...] ]) )
 *  - String (space-separated classes)
 *  - NULL / empty => empty Attribute
 */
class AttributesToolTwigExtension extends AbstractExtension {

  public function getFilters(): array {
    return [
      new TwigFilter('to_attributes', [$this, 'toAttributes']),
    ];
  }

  public function getFunctions(): array {
    return [
      new TwigFunction('to_attributes', [$this, 'toAttributes']),
    ];
  }

  /**
   * Normalize various attribute inputs into a Drupal Attribute object.
   */
  public function toAttributes($value = NULL): Attribute {
    if ($value instanceof Attribute) {
      return $value;
    }

    // String => treat as class list.
    if (is_string($value)) {
      $classes = preg_split('/\s+/', trim($value)) ?: [];
      $classes = array_values(array_filter($classes, static fn($c) => $c !== ''));
      return new Attribute($classes ? ['class' => $classes] : []);
    }

    // Sequential array => class list.
    if (is_array($value) && array_is_list($value)) {
      $classes = array_values(array_filter(array_map('strval', $value), static fn($c) => $c !== ''));
      return new Attribute($classes ? ['class' => $classes] : []);
    }

    // Associative array.
    if (is_array($value)) {
      $normalized = [];
      foreach ($value as $k => $v) {
        if ($v === NULL || $v === FALSE) { continue; }
        if ($k === 'class') {
          if ($v instanceof Attribute) { $normalized['class'] = $v; continue; }
          if (is_string($v)) {
            $v = preg_split('/\s+/', trim($v)) ?: [];
          }
          if (is_array($v)) {
            $class_list = array_values(array_filter(array_map('strval', $v), static fn($c) => $c !== ''));
            if ($class_list) { $normalized['class'] = $class_list; }
          }
          continue;
        }
        // For non-class attributes cast scalars, drop complex.
        if (is_scalar($v) || (is_object($v) && method_exists($v, '__toString'))) {
          $normalized[$k] = (string) $v;
        }
      }
      return new Attribute($normalized);
    }

    return new Attribute();
  }
}

