<?php

namespace Drupal\bootstrap_five_layouts_multiselect_test\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Layout\LayoutPluginManager;
use Drupal\Core\Render\RendererInterface;
use Drupal\bootstrap_five_layouts\BootstrapFiveLayoutsManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the multiselect test page.
 */
class MultiselectTestController extends ControllerBase {

  /**
   * The Bootstrap Five Layouts manager.
   *
   * @var \Drupal\bootstrap_five_layouts\BootstrapFiveLayoutsManager
   */
  protected $bootstrapFiveLayoutsManager;

  /**
   * The layout plugin manager.
   *
   * @var \Drupal\Core\Layout\LayoutPluginManager
   */
  protected $layoutPluginManager;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a MultiselectTestController object.
   */
  public function __construct(BootstrapFiveLayoutsManager $bootstrap_five_layouts_manager, LayoutPluginManager $layout_plugin_manager, FormBuilderInterface $form_builder, RendererInterface $renderer) {
    $this->bootstrapFiveLayoutsManager = $bootstrap_five_layouts_manager;
    $this->layoutPluginManager = $layout_plugin_manager;
    $this->formBuilder = $form_builder;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.bootstrap_five_layouts'),
      $container->get('plugin.manager.core.layout'),
      $container->get('form_builder'),
      $container->get('renderer')
    );
  }

  /**
   * Displays the multiselect test page.
   */
  public function testPage() {
    // Load the BootstrapFiveLayoutsBase (buildConfigurationForm) form on this page.
    // Get a layout plugin instance to demonstrate multiselect functionality.
    $layout_plugin = $this->layoutPluginManager->createInstance('bsfl_3col');

    // Get the default configuration for the layout.
    $default_config = $layout_plugin->defaultConfiguration();

    // Build the configuration form with default values.
    $form_state = new FormState();
    $form_state->setFormState([
      'build_info' => [
        'args' => [$default_config],
      ],
    ]);

    // Ensure form elements have proper structure for template preprocessing.
    $form = $layout_plugin->buildConfigurationForm($default_config, $form_state);
    unset($form['label']);
    unset($form['regions']);

    // Add any missing form properties that templates expect.
    $this->ensureFormElementStructure($form);

    return $form;
  }

  /**
   * Ensures form elements have proper structure for template preprocessing.
   *
   * @param array $form
   *   The form array to process.
   */
  protected function ensureFormElementStructure(array &$form) {
    foreach ($form as $key => &$element) {
      if (is_array($element) && isset($element['#type'])) {
        // Add description_display property to all form elements.
        $element['#description_display'] = 'after';

        // For textarea elements, ensure #value is set if missing.
        if ($element['#type'] === 'textarea' && !isset($element['#value'])) {
          $element['#value'] = '';
        }

        // For textfield elements, ensure #value is set if missing.
        if (in_array($element['#type'], ['textfield', 'textarea']) && !isset($element['#value'])) {
          $element['#value'] = '';
        }

        // For select elements, ensure #empty_value and #empty_option are set if missing.
        if ($element['#type'] === 'select' && !isset($element['#empty_value'])) {
          $element['#empty_value'] = '';
          if (!isset($element['#empty_option'])) {
            $element['#empty_option'] = '- ' . $this->t('None') . ' -';
          }
        }

        // Recursively process nested elements.
        if (isset($element['#type']) && in_array($element['#type'], ['container', 'details', 'fieldset'])) {
          $this->ensureFormElementStructure($element);
        }
      }
    }
  }

  /**
   * Displays the pill box test page.
   */
  public function pillBoxTest() {
    // Build and return the form.
    return $this->formBuilder()->getForm('Drupal\bootstrap_five_layouts_multiselect_test\Form\PillsTestForm');
  }

}
