<?php

namespace Drupal\bootstrap_five_layouts_viewsstyle\Plugin\views\style;

use Drupal\Core\Link;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Drupal\bootstrap_five_layouts_viewsstyle\BootstrapFiveLayoutsViewsstyle;

/**
 * Style plugin to render each item in an ordered or unordered list.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "bootstrap_five_layouts_viewsstyle_grid",
 *   title = @Translation("Bootstrap5 | Grid"),
 *   help = @Translation("Displays rows in a Bootstrap5 Grid layout"),
 *   theme = "bootstrap_five_layouts_viewsstyle_grid",
 *   theme_file = "../bootstrap_five_layouts_viewsstyle.theme.inc",
 *   display_types = {"normal"}
 * )
 */
class BootstrapFiveLayoutsViewsstyleGrid extends StylePluginBase {
  /**
   * Overrides \Drupal\views\Plugin\views\style\StylePluginBase::usesRowPlugin.
   *
   * @var bool
   */
  protected $usesRowPlugin = TRUE;

  /**
   * Overrides \Drupal\views\Plugin\views\style\StylePluginBase::usesRowClass.
   *
   * @var bool
   */
  protected $usesRowClass = TRUE;

  /**
   * Definition.
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    $options['grid_class'] = ['default' => NULL];
    foreach (BootstrapFiveLayoutsViewsstyle::getBreakpoints() as $breakpoint) {
      $breakpoint_option = "col_$breakpoint";
      $options[$breakpoint_option] = ['default' => 'none'];
    }
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $grid_link = Url::fromUri('https://getbootstrap.com/docs/5.3/layout/', [
      'attributes' => [
        'target' => '_blank',
        'rel' => 'noopener noreferrer',
      ],
    ]);
    $utility_link = Url::fromUri('https://getbootstrap.com/docs/5.3/utilities', [
      'attributes' => [
        'target' => '_blank',
        'rel' => 'noopener noreferrer',
      ],
    ]);
    $helpers_link = Url::fromUri('https://getbootstrap.com/docs/5.3/helpers', [
      'attributes' => [
        'target' => '_blank',
        'rel' => 'noopener noreferrer',
      ],
    ]);

    $form['help'] = [
      '#markup' => '<header class="bg-primary-subtle text-white py-3 px-4">
        <p>' . $this->t('The Bootstrap grid displays content in a responsive, mobile-first fluid grid. Review online documentation:') . '</p>
        <ul>
          <li>' . $this->t('@docs', ['@docs' => Link::fromTextAndUrl($this->t('Bootstrap5 Layout/Grid'), $grid_link)->toString()]) . '</li>
          <li>' . $this->t('@docs', ['@docs' => Link::fromTextAndUrl($this->t('Bootstrap5 Utility'), $utility_link)->toString()]) . '</li>
          <li>' . $this->t('@docs', ['@docs' => Link::fromTextAndUrl($this->t('Bootstrap5 Helpers'), $helpers_link)->toString()]) . '</li>
        </ul>
      </header>',
      '#weight' => -99,
    ];

    $form['grid_class'] = [
      '#title' => $this->t('Grid row custom class'),
      '#description' => $this->t('Additional classes to provide on the grid row. Separated by a space.'),
      '#type' => 'textfield',
      '#size' => '30',
      '#default_value' => $this->options['grid_class'],
      '#weight' => 1,
    ];

    $form['row_class']['#title'] = $this->t('Custom column class');
    $form['row_class']['#weight'] = 2;

    foreach (BootstrapFiveLayoutsViewsstyle::getBreakpoints() as $breakpoint) {
      $breakpoint_option = "col_$breakpoint";
      $prefix = BootstrapFiveLayoutsViewsstyle::getColumnPrefix($breakpoint);
      $form[$breakpoint_option] = [
        '#type' => 'select',
        '#title' => $this->t("@breakpoint Columns", ['@breakpoint' => mb_strtoupper($breakpoint)]),
        '#default_value' => $this->options[$breakpoint_option] ?? NULL,
        '#weight' => 3,
        '#options' => [
          'none' => $this->t('None (or inherit from previous)'),
          $prefix => $this->t('Equal'),
          $prefix . '-auto' => $this->t('Fit to content'),
        ],
      ];
      foreach ([1, 2, 3, 4, 6, 12] as $width) {
        $form[$breakpoint_option]['#options'][$prefix . "-$width"] = $this->formatPlural(12 / $width, '@width (@count column per row)', '@width (@count columns per row)', ['@width' => $width]);
      }
    }
  }

}
