<?php

namespace Drupal\bootstrap_five_layouts\Form;

use Drupal\bootstrap_five_layouts\Traits\BytesFormatTrait;
use Drupal\bootstrap_five_layouts\BootstrapFiveLayoutsManager;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Component\Utility\Bytes;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Provides the layout builder modal configuration form.
 */
class BootstrapLayoutForm extends ConfigFormBase {
  use BytesFormatTrait;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The Bootstrap Five Layouts manager.
   *
   * @var \Drupal\bootstrap_five_layouts\BootstrapFiveLayoutsManager
   */
  protected $bootstrapFiveLayoutsManager;

  /**
   * Constructs a BootstrapLayoutForm object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, ModuleHandlerInterface $module_handler, BootstrapFiveLayoutsManager $bootstrap_five_layouts_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->moduleHandler = $module_handler;
    $this->bootstrapFiveLayoutsManager = $bootstrap_five_layouts_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('module_handler'),
      $container->get('plugin.manager.bootstrap_five_layouts')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'bootstrap_five_layouts_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return ['bootstrap_five_layouts.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('bootstrap_five_layouts.settings');

    $form['key_options'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('General Options'),
    ];

    $form['key_options']['default_container'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Container'),
      '#options' => $this->bootstrapFiveLayoutsManager->getContainerOptions(),
      '#default_value' => $config->get('default_container'),
    ];

    // context:  'base' is a config translitible item.
    $form['key_options']['xs_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('X-Small Breakpoint Optgroup Label'),
      '#default_value' => $config->get('xs_label'),
      '#description' => $this->t('<q>@base</q>is the default value.', ['@base' => 'Base']),
      '#size' => 12,
      '#required' => TRUE,
      '#maxlength' => 96,
    ];

    // Get current config values and ensure they are properly typed.
    $column_appearence = $config->get('column_appearence');
    $utilities_appearence = $config->get('utilities_appearence');

    // Ensure boolean values are properly set.
    $column_appearence = $column_appearence !== NULL ? (bool) $column_appearence : TRUE;
    $utilities_appearence = $utilities_appearence !== NULL ? (bool) $utilities_appearence : FALSE;

    $form['key_options']['column_appearence'] = [
      '#type' => 'radios',
      '#options' => [
        1 => $this->t('Open'),
        0 => $this->t('Closed'),
      ],
      '#title' => $this->t('Column Details Appearence'),
      '#default_value' => $column_appearence ? 1 : 0,
    ];

    $form['key_options']['utilities_appearence'] = [
      '#type' => 'radios',
      '#options' => [
        1 => $this->t('Open'),
        0 => $this->t('Closed'),
      ],
      '#title' => $this->t('Utilities Details Appearence'),
      '#default_value' => $utilities_appearence ? 1 : 0,
    ];

    $form['key_options']['enable_ltr'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Included LTR Utilites Options'),
      '#default_value' => $config->get('enable_ltr') ?? TRUE,
      '#description' => $this->t('When enabled, Utility Options will show left-to-right specific utility controls (like text alignment and margin/padding start/end controls).'),
    ];

    $form['dev'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Internal Options'),
    ];
    $form['dev']['description_helper_selector'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Description Helper Selector'),
      '#default_value' => $config->get('description_helper_selector') ?: '.bootstrap-five-layouts-settings-tray-admin .form-item__description',
      '#description' => $this->t('CSS selector for finding description elements in the settings tray.<br> Different themes may need different selectors or adjustments to the themes templates/preprocessing to ensure a base class of some type.<br>  Prefix with <code>.bootstrap-five-layouts-settings-tray-admin</code> to contain to Layout Builder context for Bootstrap5 Layouts uses.'),
      '#size' => 60,
      '#maxlength' => 255,
    ];

    $form['options'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Theme Options'),
    ];
    $form['options']['theme_classes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Accents Sets'),
      '#default_value' => $config->get('theme_classes'),
      '#description' =>
      '<ul>' .
      '<li>' . $this->t('Basic Usage:  Use a pipe (|) charater to seperate the <b>class name</b> and <b>title</b> of the item') . '</li>' .
      '<li>' . $this->t('Example:  <code>theme--green|Green</code>') . '</li>' .
      '<li>' . $this->t('Enter one set item per Line.') . '</li>' .
      '<li>' . $this->t('Avoid extra white-space on each line.') . '</li>' .
      '<li>' . $this->t('Do not add a period(.) to class name.') . '</li>' .
      '</ul>',
      '#rows' => 12,
    ];

    $form['options']['pillbox_classes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Clases for <q>pillbox</q>  '),
      '#default_value' => $config->get('pillbox_classes'),
      '#rows' => 12,
    ];
    $description = '<ul>';
    $pillbox_raw = (string) $config->get('pillbox_classes');
    $lines = preg_split('/\r\n|\r|\n/', $pillbox_raw);
    $non_empty_lines = array_filter(array_map('trim', $lines), static function ($line) {
      return $line !== '';
    });
    $total = count($non_empty_lines);
    if ($total > 0) {
      $description .= '<li><b>' . $this->formatPlural(
        $total,
        '@count item currently stored.',
        '@count items currently stored.'
      ) . '</b></li>';
    }
    $description .= '<li>' . $this->t('Items will be filtered for duplicates.') .
        '<li>' . $this->t('Multiples per line will be split into multiple lines.') . '</li>' .
        '<li>' . $this->t('Do not add a period(.) to class name.') . '</li>' .
        '</ul>';
    $form['options']['pillbox_classes']['#description'] = $description;
    // Intentionally left blank placeholder removed.
    $form['file'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Layout Background Integration'),
      // Early dev @scope..
      '#access' => FALSE,
    ];
    $form['file']['enable_background'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Background Feature'),
      '#default_value' => $config->get('enable_background') ?? FALSE,
    ];
    $form['file']['background'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="enable_background"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['file']['background']['allow_media'] = [
      '#type' => 'radios',
      '#options' => ['media' => $this->t('Media'), 'file' => $this->t('Image/File')],
      '#title' => $this->t('File Type Preference'),
      '#default_value' => $config->get('allow_media'),
      '#field_prefix' => $this->t('Warning:  Avoid changes while in a production enviorment.'),
    ];

    if ($this->moduleHandler->moduleExists('file')) {
      // Get PHP upload_max_filesize and convert to bytes.
      $php_upload_max = ini_get('upload_max_filesize');
      $php_upload_max_bytes = $this->getPhpUploadMaxBytes($php_upload_max);

      $form['file']['background']['max_file_size'] = [
        '#type' => 'textfield',
        '#maxlength' => '56',
        '#size' => '19',
        '#title' => $this->t('Maximum File Size'),
        '#field_suffix' => '<small>' . $this->t('Bytes (@current)', [
          '@current' => $this->formatBytes(($config->get('max_file_size') ?: $php_upload_max_bytes)),
        ]) . '</small>',
        '#min' => 1,
        '#step' => 1,
        '#default_value' => $config->get('max_file_size') ?: $php_upload_max_bytes,
        '#description' => '<ul>' .
        '<li>' . $this->t('Set the maximum file size (in bytes) for uploads.') . '</li>' .
        '<li>' . $this->t('Auto Convert Helper:  Input will convert formatted size to bytes.') . '</li>' .
        '<li>' . $this->t('Current PHP max: @php_max (@php_max_bytes bytes).', [
          '@php_max' => $php_upload_max,
          '@php_max_bytes' => $php_upload_max_bytes,
        ]) . '</li>' .
        '<li>' . $this->t('Minimum allowed file size is 20 bytes.') . '</li>' .
        '</ul>',
        '#states' => [
          'visible' => [
            ':input[name="allow_media"]' => ['value' => 'file'],
          ],
        ],
      ];
    }
    else {
      $form['file']['background']['allow_media']['#access'] = FALSE;
    }

    $form['file']['background']['allowed_file_types'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Allowed Image File Types'),
      '#maxlength' => '256',
      '#size' => '19',
      '#default_value' => $config->get('allowed_file_types'),
      '#states' => [
        'visible' => [
          ':input[name="allow_media"]' => ['value' => 'file'],
        ],
      ],
    ];
    // in-case media is offline, just carry the values..
    $form['file']['background']['allowed_media_types'] = [
      '#title' => $this->t('Allowed Media Types'),
      '#type' => 'checkboxes',
      '#options' => $config->get('allowed_media_types') ?? ['file'],
      '#default_value' => $config->get('allowed_media_types') ?? ['file'],
      '#states' => [
        'visible' => [
          ':input[name="allow_media"]' => ['value' => 'media'],
        ],
      ],
    ];

    if ($this->moduleHandler->moduleExists('media_library_form_element')) {
      // Load the currently active media types into a checkbox input.
      $media_type_storage = \Drupal::entityTypeManager()->getStorage('media_type');
      $media_types = $media_type_storage->loadMultiple();
      $media_type_options = [];
      foreach ($media_types as $media_type) {
        $media_type_options[$media_type->id()] = $media_type->label();
      }
      $form['file']['background']['allowed_media_types']['#options'] = $media_type_options;
    }
    else {
      unset($form['file']['background']['allow_media']['#options']['media']);
      $link = Link::fromTextAndUrl('Media Library Form Element', Url::fromUri('https://www.drupal.org/project/media_library_form_element', ['attributes' => ['target' => '_blank']]));
      $form['file']['background']['allow_media']['#description'] = $this->t('Module @link is not enabled at this time.', ['@link' => $link->toString()]);
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $theme_classes = $form_state->getValue('theme_classes');
    $lines = preg_split('/\r\n|\r|\n/', $theme_classes);
    foreach ($lines as $index => $line) {
      $line = trim($line);
      if ($line === '') {
        // Skip empty lines.
        continue;
      }
      $parts = explode('|', $line);
      if (count($parts) !== 2) {
        $form_state->setErrorByName('theme_classes', $this->t('Error on line @line: <q>@content</q>. Each line must contain exactly one pipe (|).', ['@line' => $index + 1, '@content' => $line]));
        return;
      }
      $left = $parts[0];
      $right = $parts[1];
      if (strlen($left) < 2 || strlen(trim($right)) < 2) {
        $form_state->setErrorByName('theme_classes', $this->t('Error on line @line: <q>@content</q>.  Each segment must be at least 2 characters.', ['@line' => $index + 1, '@content' => $line]));
        return;
      }
      if (preg_match('/\s/', $left)) {
        $form_state->setErrorByName('theme_classes', $this->t('Error on line @line: <q>@content</q>.  No whitespace is allowed in the class name segment.', ['@line' => $index + 1, '@content' => $line]));
        return;
      }
      if (strpos($left, '.') === 0) {
        $form_state->setErrorByName('theme_classes', $this->t('Error on line @line: <q>@content</q>.  The class name segment must not start with a (.) period.', ['@line' => $index + 1, '@content' => $line]));
        return;
      }
      if ($right !== trim($right)) {
        $form_state->setErrorByName('theme_classes', $this->t('Error on line @line: <q>@content</q>.  The title segment cannot start or end with whitespace.', ['@line' => $index + 1, '@content' => $line]));
        return;
      }
    }

    // Validate and normalize pillbox_classes: split multiple classes per line,
    // remove duplicates, trim whitespace, and ensure valid CSS class tokens.
    $pillbox_raw = (string) $form_state->getValue('pillbox_classes');
    if ($pillbox_raw !== '') {
      $pillbox_lines = preg_split('/\r\n|\r|\n/', $pillbox_raw);
      $normalized_tokens = [];
      $seen = [];
      foreach ($pillbox_lines as $line_index => $raw_line) {
        $raw_line = trim($raw_line);
        if ($raw_line === '') {
          continue;
        }
        // Split on any whitespace to allow multiple classes per line.
        $tokens = preg_split('/\s+/', $raw_line);
        foreach ($tokens as $token) {
          $token = trim($token);
          if ($token === '') {
            continue;
          }
          // Reject tokens starting with '.' and any invalid CSS class name chars.
          if (strpos($token, '.') === 0) {
            $form_state->setErrorByName('pillbox_classes', $this->t('Invalid token <q><code>@token</code></q> on line @line: do not prefix class names with a period.', ['@token' => $token, '@line' => $line_index + 1]));
            return;
          }
          // CSS class name pattern (simplified): start with letter/_/-, then letters/digits/_/-.
          if (!preg_match('/^[A-Za-z_-][A-Za-z0-9_-]*$/', $token)) {
            $form_state->setErrorByName('pillbox_classes', $this->t('Invalid class name <q><code>@token</code></q> on line @line.', ['@token' => $token, '@line' => $line_index + 1]));
            return;
          }
          $key = mb_strtolower($token);
          if (!isset($seen[$key])) {
            $seen[$key] = TRUE;
            $normalized_tokens[] = $token;
          }
        }
      }
      // Set normalized value back: one class per line.
      $form_state->setValue('pillbox_classes', implode("\n", $normalized_tokens));
    }

    // Validate max_file_size.
    $max_file_size = $this->convertToBytes($form_state->getValue('max_file_size'));
    if ($max_file_size === FALSE || (!is_numeric($max_file_size) && $max_file_size > 0)) {
      $form_state->setErrorByName('max_file_size', $this->t('Can not convert input'));
    }

    // @convert helper.  Set the converted site back to the input.
    if ($max_file_size !== NULL && is_numeric($max_file_size) && $max_file_size > 0) {
      $form_state->setValue('max_file_size', $max_file_size);
    }

    if ($max_file_size !== NULL && (!is_numeric($max_file_size) || $max_file_size < 1)) {
      $form_state->setErrorByName('max_file_size', $this->t('Maximum file size must be a positive number.'));
    }
    if ($max_file_size !== NULL && (!is_numeric($max_file_size) || $max_file_size < 20)) {
      $form_state->setErrorByName('max_file_size', $this->t('Sorry, the file’s header size would be too small.  The minimum expected file size is 20 bytes&hellip;'));
    }
    elseif ($max_file_size !== NULL) {
      $php_upload_max = $this->convertToBytes(ini_get('upload_max_filesize'));
      if ($max_file_size > $php_upload_max) {
        $form_state->setErrorByName('max_file_size', $this->t('Maximum file size cannot exceed the PHP <code>upload_max_filesize</code> setting (@php_max_bytes bytes).', ['@php_max_bytes' => $php_upload_max]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('bootstrap_five_layouts.settings')
      ->set('default_container', $form_state->getValue('default_container'))
      ->set('utilities_appearence', (bool) $form_state->getValue('utilities_appearence'))
      ->set('column_appearence', (bool) $form_state->getValue('column_appearence'))
      ->set('theme_classes', $form_state->getValue('theme_classes'))
      ->set('pillbox_classes', $form_state->getValue('pillbox_classes'))
      ->set('xs_label', $form_state->getValue('xs_label'))
      ->set('description_helper_selector', $form_state->getValue('description_helper_selector'))
      ->set('enable_ltr', $form_state->getValue('enable_ltr'))
      ->set('allowed_media_types', array_filter($form_state->getValue('allowed_media_types', [])))
      ->set('max_file_size', $form_state->getValue('max_file_size'))
      ->set('allowed_file_types', $form_state->getValue('allowed_file_types'))
      ->set('allow_media', $form_state->getValue('allow_media'))
      ->set('enable_background', $form_state->getValue('enable_background'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Gets the PHP upload_max_filesize value in bytes.
   *
   * @param string $php_upload_max
   *   The PHP upload_max_filesize value.
   *
   * @return int
   *   The maximum upload file size in bytes.
   */
  private function getPhpUploadMaxBytes($php_upload_max = 0) {
    return Bytes::toNumber($php_upload_max);
  }

}
