<?php

namespace Drupal\brandfolder\Controller;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\brandfolder\Event\BrandfolderWebhookEvent;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\Access\AccessInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Component\Serialization\Json;

/**
 * Handle incoming Brandfolder webhook requests/events.
 */
class IncomingWebhookController extends ControllerBase implements AccessInterface {

  /**
   * Valid event types.
   *
   * @var array $valid_event_types
   */
  protected $valid_event_types = [
    'asset.create',
    'asset.update',
    'asset.delete',
  ];

  /**
   * Access management. Block requests that are blatantly invalid.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Request $request): AccessResultInterface {
    $valid_payload = FALSE;
    $payload = json_decode($request->getContent(), TRUE);
    if (isset($payload['data']['attributes'])) {
      $data = $payload['data']['attributes'];
      $event_type = $data['event_type'];
      if (in_array($event_type, $this->valid_event_types)) {
        if (isset($data['key']) && is_string($data['key']) && strlen($data['key']) > 0) {
          $valid_payload = TRUE;
        }
      }
    }

    return AccessResult::allowedIf($valid_payload);
  }

  /**
   * Handle incoming webhooks.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *
   * @return \Symfony\Component\HttpFoundation\Response
   */
  public function webhookListener(Request $request) {
    $payload = json_decode($request->getContent(), TRUE);
    $is_request_valid = isset($payload['data']['attributes']);

    $bf_config = \Drupal::config('brandfolder.settings');
    if ($bf_config->get('verbose_log_mode')) {
      $logger = \Drupal::logger('brandfolder');
      $logger->debug('Brandfolder webhook received. It looks like {validity} request. Payload: {payload}', [
        'validity' => $is_request_valid ? 'a valid' : 'an invalid',
        'payload' => Json::encode($payload),
      ]);
    }

    if ($is_request_valid) {
      $bf_event_type = $payload['data']['attributes']['event_type'];
      // Fire/dispatch events so Drupal modules can act on this webhook.
      $dispatcher = \Drupal::service('event_dispatcher');
      $dispatcher->dispatch(new BrandfolderWebhookEvent($payload['data']), $bf_event_type);
    }

    return new Response($this->t('Webhook handling complete.'), Response::HTTP_OK);
  }

}
