<?php

declare(strict_types=1);

namespace Drupal\brightcove_proxy\EventSubscriber;

use Drupal\brightcove\Event\HttpClientConfigEvent;
use Drupal\Core\Config\ConfigFactory;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to the HTTP client proxy config event.
 */
final readonly class HttpClientProxyConfigEventSubscriber implements EventSubscriberInterface {

  /**
   * Initializes a Drupal initializes event subscriber.
   *
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   Config factory.
   */
  public function __construct(
    private ConfigFactory $configFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      HttpClientConfigEvent::class => 'configureBrightcoveClientProxy',
    ];
  }

  /**
   * Configure Brightcove client proxy.
   *
   * @param \Drupal\brightcove\Event\HttpClientConfigEvent $event
   *   HTTP client config event.
   */
  public function configureBrightcoveClientProxy(HttpClientConfigEvent $event): void {
    $config = $this->configFactory->get('brightcove_proxy.config');

    // Initialize proxy config for Brightcove client if enabled.
    if ($config->get('use_proxy')) {
      $event->set('curl', [
        \CURLOPT_PROXYUSERPWD => "{$config->get('proxy_username')}:{$config->get('proxy_password')}",
        \CURLOPT_HTTPAUTH => $config->get('proxy_auth'),
        \CURLOPT_PROXYTYPE => $config->get('proxy_type'),
        \CURLOPT_PROXY => $config->get('proxy'),
        \CURLOPT_PROXYPORT => $config->get('proxy_port'),
        \CURLOPT_HTTPPROXYTUNNEL => $config->get('http_proxy_tunnel'),
      ]);
    }
  }

}
