<?php

declare(strict_types=1);

namespace Drupal\brightcove;

use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\Entity\Storage\PlayerStorageInterface;
use Drupal\brightcove\Services\ApiClientHandlerInterface;
use Drupal\brightcove\Type\ApiClientStatusType;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a listing of Brightcove API Client entities.
 */
class ApiClientListBuilder extends ConfigEntityListBuilder {

  /**
   * Initializes a new API Client list builder.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entityType
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param string $defaultAPIClient
   *   The ID of the default API client.
   * @param \Drupal\brightcove\Entity\Storage\PlayerStorageInterface $playerStorage
   *   Player storage.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   String translation.
   * @param \Drupal\brightcove\Services\ApiClientHandlerInterface $apiClientHandler
   *   API client handler.
   */
  public function __construct(
    EntityTypeInterface $entityType,
    EntityStorageInterface $storage,
    protected readonly string $defaultAPIClient,
    protected readonly PlayerStorageInterface $playerStorage,
    TranslationInterface $stringTranslation,
    protected readonly ApiClientHandlerInterface $apiClientHandler,
  ) {
    parent::__construct($entityType, $storage);
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): static {
    $entity_type_manager = $container->get('entity_type.manager');

    return new static(
      $entity_type,
      $entity_type_manager->getStorage($entity_type->id()),
      $container->get('brightcove.settings')->getDefaultApiClientId(),
      $entity_type_manager->getStorage('brightcove_player'),
      $container->get('string_translation'),
      $container->get('brightcove.api.client'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    return [
      'label' => $this->t('Brightcove API Client'),
      'id' => $this->t('Machine name'),
      'account_id' => $this->t('Account ID'),
      'default_player' => $this->t('Default player'),
      'client_status' => $this->t('Status'),
    ] + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    assert($entity instanceof ApiClientInterface);

    $api_client_id = $entity->id();
    $row = [
      'label' => $entity->label(),
      'id' => $api_client_id,
      'account_id' => $entity->getAccountId(),
      'default_player' => $this->playerStorage->getList([$api_client_id])[$entity->getDefaultPlayer()],
    ];

    if ($this->defaultAPIClient === $api_client_id) {
      $row['id'] .= ' ' . $this->t('(default)');
    }

    // Test API client connection.
    $row['client_status'] = match($this->apiClientHandler->testConnection($entity)->status) {
      ApiClientStatusType::OK => $this->t('OK'),
      ApiClientStatusType::ERROR => $this->t('Error'),
      ApiClientStatusType::INITIALIZING => $this->t('Initializing'),
    };

    return $row + parent::buildRow($entity);
  }

}
