<?php

declare(strict_types=1);

namespace Drupal\brightcove;

use Brightcove\API\ApiClientInterface as BrightcoveApiClientInterface;
use Brightcove\API\CMS;
use Brightcove\API\DI;
use Brightcove\API\Exception\ApiException;
use Brightcove\API\PM;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\Entity\VideoInterface;
use Drupal\brightcove\Entity\Player;
use Drupal\brightcove\Entity\Playlist;
use Drupal\brightcove\Entity\Video;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Url;
use Drupal\taxonomy\Entity\Term;

/**
 * Utility class for Brightcove.
 */
class BrightcoveUtil {

  /**
   * Array of BrightcoveAPIClient objects.
   *
   * @var \Drupal\brightcove\Entity\ApiClient[]
   */
  protected static $apiClients = [];

  /**
   * Array of CMS objects.
   *
   * @var \Brightcove\API\CMS[]
   */
  protected static $cmsApis = [];

  /**
   * Array of DI objects.
   *
   * @var \Brightcove\API\DI[]
   */
  protected static $diApis = [];

  /**
   * Array of PM objects.
   *
   * @var \Brightcove\API\PM[]
   */
  protected static $pmApis = [];

  /**
   * Gets BrightcoveAPIClient entity.
   *
   * @param string $entity_id
   *   The entity ID of the BrightcoveAPIClient.
   *
   * @return \Drupal\brightcove\Entity\ApiClientInterface
   *   Loaded BrightcoveAPIClient object.
   */
  public static function getApiClient(string $entity_id): ApiClientInterface {
    // Load BrightcoveAPIClient if it wasn't already.
    if (!isset(self::$apiClients[$entity_id])) {
      self::$apiClients[$entity_id] = \Drupal::getContainer()->get('entity_type.manager')->getStorage('brightcove_api_client')->load($entity_id);
    }

    return self::$apiClients[$entity_id];
  }

  /**
   * Gets Brightcove client.
   *
   * @param string $entity_id
   *   BrightcoveAPIClient entity ID.
   *
   * @return \Brightcove\API\Client
   *   Loaded Brightcove client.
   */
  public static function getClient(string $entity_id): BrightcoveApiClientInterface {
    return \Drupal::getContainer()->get('brightcove.api.client')
      ->getClient(self::getApiClient($entity_id));
  }

  /**
   * Gets Brightcove CMS API.
   *
   * @param string $entity_id
   *   BrightcoveAPIClient entity ID.
   *
   * @return \Brightcove\API\CMS
   *   Initialized Brightcove CMS API.
   */
  public static function getCmsApi(string $entity_id): CMS {
    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = self::getApiClient($entity_id);

    // Create new \Brightcove\API\CMS object if it is not exists yet.
    if (!isset(self::$cmsApis[$entity_id]) || !\Drupal::getContainer()->get('brightcove.expirable_access_token_storage')->has($api_client->getClientId())) {
      self::$cmsApis[$entity_id] = new CMS(self::getClient($entity_id), self::$apiClients[$entity_id]->getAccountId());
    }

    return self::$cmsApis[$entity_id];
  }

  /**
   * Gets Brightcove DI API.
   *
   * @param string $entity_id
   *   BrightcoveAPIClient entity ID.
   *
   * @return \Brightcove\API\DI
   *   Initialized Brightcove CMS API.
   */
  public static function getDiApi(string $entity_id): DI {
    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = self::getApiClient($entity_id);

    // Create new \Brightcove\API\DI object if it is not exists yet.
    if (!isset(self::$diApis[$entity_id]) || !\Drupal::getContainer()->get('brightcove.expirable_access_token_storage')->has($api_client->getClientId())) {
      self::$diApis[$entity_id] = new DI(self::getClient($entity_id), self::$apiClients[$entity_id]->getAccountId());
    }

    return self::$diApis[$entity_id];
  }

  /**
   * Gets Brightcove PM API.
   *
   * @param string $entity_id
   *   BrightcoveAPIClient entity ID.
   *
   * @return \Brightcove\API\PM
   *   Initialized Brightcove PM API.
   */
  public static function getPmApi(string $entity_id): PM {
    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = self::getApiClient($entity_id);

    // Create new \Brightcove\API\PM object if it is not exists yet.
    if (!isset(self::$pmApis[$entity_id]) || !\Drupal::getContainer()->get('brightcove.expirable_access_token_storage')->has($api_client->getClientId())) {
      self::$pmApis[$entity_id] = new PM(self::getClient($entity_id), self::$apiClients[$entity_id]->getAccountId());
    }

    return self::$pmApis[$entity_id];
  }

  /**
   * Check updated version of the CMS entity.
   *
   * If the checked CMS entity has a newer version of it on Brightcove then
   * show a message about it with a link to be able to update the local
   * version.
   *
   * @param \Drupal\brightcove\CmsEntityInterface $entity
   *   Brightcove CMS Entity, can be BrightcoveVideo or BrightcovePlaylist.
   *   Player is currently not supported.
   *
   * @throws \Exception
   *   If the version for the given entity is cannot be checked.
   */
  public static function checkUpdatedVersion(CmsEntityInterface $entity): void {
    $client = self::getClient($entity->getApiClient());

    if (!is_null($client)) {
      $cms = self::getCmsApi($entity->getApiClient());

      try {
        // Check entity type.
        if ($entity instanceof Video) {
          $entity_type = 'video';
        }
        elseif ($entity instanceof Playlist) {
          $entity_type = 'playlist';
        }
        else {
          throw new \Exception(strtr('Cannot check version for :entity_type entity.', [
            ':entity_type' => get_class($entity),
          ]));
        }

        // Validate brightcove ID.
        $brightcove_id = $entity->getBrightcoveId();
        if ($brightcove_id === NULL) {
          throw new ApiException();
        }

        // Get list of entities based on their type.
        if ($entity instanceof Video) {
          $cms_entity = $cms->getVideo($brightcove_id);
        }
        elseif ($entity instanceof Playlist) {
          $cms_entity = $cms->getPlaylist($brightcove_id);
        }

        $updated_at = $cms_entity->getUpdatedAt();
        if ($entity->getChangedTime() < ($updated_at !== NULL ? strtotime($updated_at) : 0)) {
          $url = Url::fromRoute("brightcove_manual_update_{$entity_type}", ['entity_id' => $entity->id()], ['query' => ['token' => \Drupal::getContainer()->get('csrf_token')->get("brightcove_{$entity_type}/{$entity->id()}/update")]]);

          \Drupal::messenger()->addWarning(t("There is a newer version of this :type on Brightcove, you may want to <a href=':url'>update the local version</a> before editing it.", [
            ':type' => $entity_type,
            ':url' => $url->toString(),
          ]));
        }
      }
      catch (ApiException) {
        $url = Url::fromRoute("entity.brightcove_{$entity_type}.delete_form", ["brightcove_{$entity_type}" => $entity->id()]);
        \Drupal::messenger()->addError(t("This :type no longer exists on Brightcove. You may want to <a href=':url'>delete the local version</a> too.", [
          ':type' => $entity_type,
          ':url' => $url->toString(),
        ]));
      }
    }
    else {
      \Drupal::messenger()->addError(t('Brightcove API connection error: :error', [
        ':error' => self::getApiClient($entity->getApiClient())->getClientStatusMessage(),
      ]));
    }
  }

  /**
   * Helper function to get default player for the given entity.
   *
   * @param \Drupal\brightcove\VideoPlaylistCmsEntityInterface $entity
   *   Video or playlist entity.
   *
   * @return string
   *   The ID of the player.
   */
  public static function getDefaultPlayer(VideoPlaylistCmsEntityInterface $entity): string {
    if ($player = $entity->getPlayer()) {
      return Player::load($player)->getPlayerId();
    }

    $api_client = self::getApiClient($entity->getApiClient());
    return $api_client->getDefaultPlayer();
  }

  /**
   * Helper function to save or update tags.
   *
   * @param \Drupal\brightcove\VideoPlaylistCmsEntityInterface $entity
   *   Video or playlist entity.
   * @param string $api_client_id
   *   API Client ID.
   * @param array $tags
   *   The list of tags from brightcove.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public static function saveOrUpdateTags(VideoPlaylistCmsEntityInterface $entity, string $api_client_id, array $tags = []): void {
    $entity_tags = [];
    $video_entity_tags = $entity->getTags();
    foreach ($video_entity_tags as $index => $tag) {
      /** @var \Drupal\taxonomy\Entity\Term $term */
      $term = Term::load($tag['target_id']);
      if (!is_null($term)) {
        $entity_tags[$term->id()] = $term->getName();
      }
      // Remove non-existing tag references from the video, if there would
      // be any.
      else {
        unset($video_entity_tags[$index]);
        $entity->setTags($video_entity_tags);
      }
    }
    if (array_values($entity_tags) !== $tags) {
      // Remove deleted tags from the video.
      if (!empty($entity->id())) {
        $tags_to_remove = array_diff($entity_tags, $tags);
        foreach (array_keys($tags_to_remove) as $entity_id) {
          unset($entity_tags[$entity_id]);
        }
      }

      // Add new tags.
      $new_tags = array_diff($tags, $entity_tags);
      $entity_tags = array_keys($entity_tags);
      foreach ($new_tags as $tag) {
        $taxonomy_term = NULL;
        $existing_tags = \Drupal::entityQuery('taxonomy_term')
          ->condition('vid', VideoInterface::TAGS_VID)
          ->condition('name', $tag)
          ->accessCheck()
          ->execute();

        // Create new Taxonomy term item.
        if (empty($existing_tags)) {
          $values = [
            'name' => $tag,
            'vid' => VideoInterface::TAGS_VID,
            'brightcove_api_client' => [
              'target_id' => $api_client_id,
            ],
          ];
          $taxonomy_term = Term::create($values);
          $taxonomy_term->save();
        }
        $entity_tags[] = isset($taxonomy_term) ? $taxonomy_term->id() : reset($existing_tags);
      }
      $entity->setTags($entity_tags);
    }
  }

  /**
   * Returns the absolute URL path for the notification callback.
   *
   * @return string
   *   The absolute URL path for the notification callback.
   */
  public static function getDefaultSubscriptionUrl(): string {
    return Url::fromRoute('brightcove_notification_callback', [], ['absolute' => TRUE])->toString();
  }

  /**
   * Run a piece of code with semaphore check.
   *
   * @param callable $function
   *   Function that needs to be run in sync.
   * @param \Drupal\Core\Lock\LockBackendInterface|null $lock
   *   Lock backend.
   *
   * @return bool|mixed
   *   FALSE if the execution was failed, otherwise it will return what the
   *   callable function returned.
   */
  public static function runWithSemaphore(callable $function, ?LockBackendInterface $lock = NULL): mixed {
    $lock_name = 'brightcove_semaphore';
    try {
      // Make sure that the lock service is available.
      if ($lock === NULL) {
        $lock = \Drupal::lock();
      }

      // Basic semaphore to prevent race conditions, this is needed because
      // Brightcove may call callbacks again before the previous one would
      // finish.
      //
      // To make sure that the waiting doesn't run indefinitely limit the
      // maximum iterations to 600 cycles, which in worst case scenario would
      // mean 5 minutes maximum wait time.
      $limit = 600;
      for ($i = 0; $i < $limit; $i++) {
        // Try to acquire lock.
        for (; $i < $limit && !$lock->lockMayBeAvailable($lock_name); $i++) {
          // Wait random time between 100 and 500 milliseconds on each try.
          // The lock backend's wait() method is not used here as it can wait
          // only seconds that may take too long.
          usleep(mt_rand(100000, 500000));
        }

        // Make sure that other processes have not acquired the lock while we
        // waited then try to acquire lock as soon as we can.
        if ($lock->lockMayBeAvailable($lock_name) && $lock->acquire($lock_name)) {
          break;
        }
      }

      // If we couldn't acquire the lock in the given time, release the lock
      // (finally block will take care of it) and return with FALSE.
      if (600 <= $i) {
        return FALSE;
      }

      // Run function.
      return $function();
    }
    catch (\Exception $e) {
      // Log error, and return with FALSE.
      \Drupal::getContainer()
        ->get('brightcove.logger')
        ->logException($e, 'Running code with semaphore failed.');
      return FALSE;
    }
    finally {
      // Release semaphore.
      // This will always run regardless what happened.
      $lock->release($lock_name);
    }
  }

}
