<?php

declare(strict_types=1);

namespace Drupal\brightcove\Controller;

use Drupal\brightcove\Services\CmsApiHandlerInterface;
use Drupal\brightcove\Services\PlaylistHelperInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides controller for playlist update.
 */
final class PlaylistUpdateController extends ControllerBase {

  /**
   * Controller constructor.
   *
   * @param \Drupal\brightcove\Services\CmsApiHandlerInterface $cmsApiHandler
   *   CMS API handler.
   * @param \Drupal\brightcove\Services\PlaylistHelperInterface $playlistHelper
   *   Playlist helper.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    private readonly CmsApiHandlerInterface $cmsApiHandler,
    private readonly PlaylistHelperInterface $playlistHelper,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('brightcove.api.cms'),
      $container->get('brightcove.playlist.helper'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Callback to update the existing Playlist with the latest version.
   *
   * @param int $entity_id
   *   The ID of the playlist in Drupal.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirection response.
   *
   * @throws \Exception
   *   If BrightcoveAPIClient ID is missing when a new entity is being created.
   */
  public function __invoke(int $entity_id): Response {
    /** @var \Drupal\brightcove\Entity\Playlist $playlist */
    $playlist = $this->entityTypeManager->getStorage('brightcove_playlist')->load($entity_id);
    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = $this->entityTypeManager->getStorage('brightcove_api_client')->load($playlist->getApiClient());
    $cms_api = $this->cmsApiHandler->getApi($api_client);

    // Update playlist.
    $this->playlistHelper->createOrUpdate($cms_api->getPlaylist($playlist->getBrightcoveId()), $api_client->id());

    // Redirect back to the playlist edit form.
    return $this->redirect('entity.brightcove_playlist.edit_form', [
      'brightcove_playlist' => $entity_id,
    ]);
  }

}
