<?php

declare(strict_types=1);

namespace Drupal\brightcove\Controller;

use Brightcove\API\Exception\ApiException;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\brightcove\Entity\Subscription;
use Drupal\brightcove\Entity\TextTrack;
use Drupal\brightcove\Entity\Exception\SubscriptionException;
use Drupal\brightcove\Services\CmsApiHandlerInterface;
use Drupal\brightcove\Services\IngestionInterface;
use Drupal\brightcove\Services\LoggerInterface;
use Drupal\brightcove\Services\VideoHelperInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\Core\Utility\LinkGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides controller for subscription related callbacks.
 */
final class SubscriptionController extends ControllerBase {

  /**
   * Initializes a subscription controller.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   Database connection.
   * @param \Drupal\Core\Utility\LinkGeneratorInterface $linkGenerator
   *   Link generator.
   * @param \Drupal\Core\Lock\LockBackendInterface $lock
   *   Lock backend.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   String translation.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger.
   * @param \Drupal\brightcove\Services\LoggerInterface $logger
   *   Logger.
   * @param \Drupal\brightcove\Services\IngestionInterface $ingestion
   *   Ingestion.
   * @param \Drupal\brightcove\Services\VideoHelperInterface $videoHelper
   *   Video helper.
   * @param \Drupal\brightcove\Services\CmsApiHandlerInterface $cmsApiHandler
   *   CMS API handler.
   */
  public function __construct(
    private readonly Connection $connection,
    private readonly LinkGeneratorInterface $linkGenerator,
    private readonly LockBackendInterface $lock,
    TranslationInterface $stringTranslation,
    EntityTypeManagerInterface $entityTypeManager,
    MessengerInterface $messenger,
    private readonly LoggerInterface $logger,
    private readonly IngestionInterface $ingestion,
    private readonly VideoHelperInterface $videoHelper,
    private readonly CmsApiHandlerInterface $cmsApiHandler,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->stringTranslation = $stringTranslation;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('database'),
      $container->get('link_generator'),
      $container->get('lock'),
      $container->get('string_translation'),
      $container->get('entity_type.manager'),
      $container->get('messenger'),
      $container->get('brightcove.logger'),
      $container->get('brightcove.ingestion'),
      $container->get('brightcove.video_helper'),
      $container->get('brightcove.api.cms'),
    );
  }

  /**
   * Menu callback to handle the Brightcove notification callback.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   *
   * @throws \Exception
   */
  public function notificationCallback(Request $request): Response {
    // Immediately return with a success response, then continue the processing
    // of the notification.
    $response = new Response('', Response::HTTP_NO_CONTENT);
    $response->send();

    $this->logger->debug('notification callback');

    // Ensure that the requests does not run in parallel as it could update the
    // same entity causing inconsistent state.
    BrightcoveUtil::runWithSemaphore(function () use ($request): void {
      $content = json_decode($request->getContent(), TRUE);

      switch ($content['event']) {
        case 'video-change':
          // Try to update an existing video or create a new one if not exist.
          try {
            // Get CMS API.
            /** @var \Drupal\brightcove\Entity\ApiClientInterface[] $api_client */
            $api_clients = $this->entityTypeManager->getStorage('brightcove_api_client')->loadByProperties([
              'account_id' => $content['account_id'],
            ]);
            if (!empty($api_clients)) {
              // Create or update Video.
              $api_client = reset($api_clients);
              $cms_api = $this->cmsApiHandler->getApi($api_client);
              $video = $cms_api->getVideo($content['video']);
              $video_entity = $this->videoHelper->createOrUpdate($video, $api_client->id());

              // Create, update or delete Text Track.
              if (!$this->ingestion->isFieldMarkedForIngestion($video_entity, 'text_tracks')) {
                $text_track_storage = $this->entityTypeManager->getStorage('brightcove_text_track');

                // Gather Text Track IDs.
                $text_tracks = $video_entity->getTextTracks();
                $ids = [];
                foreach ($text_tracks as $text_track) {
                  $ids[] = $text_track['target_id'];
                }

                $text_tracks_to_delete = [];
                foreach ($text_track_storage->loadMultiple($ids) as $text_track) {
                  $text_tracks_to_delete[$text_track->getTextTrackId()] = $text_track;
                }

                foreach ($video->getTextTracks() as $text_track) {
                  // Remove existing entities from the deletion list.
                  $id = $text_track->getId();
                  if (isset($text_tracks_to_delete[$id])) {
                    unset($text_tracks_to_delete[$id]);
                  }

                  TextTrack::createOrUpdate($text_track, $text_track_storage, $video_entity->id());
                }

                if (!empty($text_tracks_to_delete)) {
                  $text_track_storage->delete($text_tracks_to_delete);
                }
              }
            }
          }
          catch (ApiException $e) {
            // In case of 404 we have nothing to do.
            if ($e->getCode() !== 404) {
              $this->logger->logException($e, 'Failed to update video via the notification callback.');
            }
          }
          break;

        case 'master-video-change':
          // @todo Add handler.
          break;
      }
    }, $this->lock);

    return new Response(status: Response::HTTP_NO_CONTENT);
  }

  /**
   * Lists available Brightcove Subscriptions.
   *
   * @return array
   *   Renderable page elements.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function listSubscriptions(): array {
    // Set headers.
    $header = [
      'endpoint' => $this->t('Endpoint'),
      'api_client' => $this->t('API Client'),
      'events' => $this->t('Events'),
      'operations' => $this->t('Operations'),
    ];

    // Get Subscriptions.
    $brightcove_subscriptions = Subscription::loadMultiple();

    // Whether a warning has been shown about the missing subscriptions on
    // Brightcove or not.
    $warning_set = FALSE;

    // Assemble subscription list.
    $rows = [];
    foreach ($brightcove_subscriptions as $key => $brightcove_subscription) {
      $api_client = $brightcove_subscription->getApiClient();

      $rows[$key] = [
        'endpoint' => $brightcove_subscription->getEndpoint() . ($brightcove_subscription->isDefault() ? " ({$this->t('default')})" : ''),
        'api_client' => !empty($api_client) ? $this->linkGenerator->generate($api_client->label(), Url::fromRoute('entity.brightcove_api_client.edit_form', [
          'brightcove_api_client' => $api_client->id(),
        ])) : '',
        'events' => implode(', ', array_filter($brightcove_subscription->getEvents(), function ($value) {
          return !empty($value);
        })),
      ];

      // Default subscriptions can be enabled or disabled only.
      if ($brightcove_subscription->isDefault()) {
        $enable_link = Url::fromRoute('entity.brightcove_subscription.enable', [
          'id' => $brightcove_subscription->getId(),
        ]);

        $disable_link = Url::fromRoute('entity.brightcove_subscription.disable', [
          'id' => $brightcove_subscription->getId(),
        ]);

        $rows[$key]['operations'] = [
          'data' => [
            '#type' => 'operations',
            '#links' => [
              'change_status' => [
                'title' => $brightcove_subscription->isActive() ? $this->t('Disable') : $this->t('Enable'),
                'url' => $brightcove_subscription->isActive() ? $disable_link : $enable_link,
              ],
            ],
          ],
        ];
      }
      // Otherwise, show delete button or create button as well if needed.
      else {
        $subscriptions = Subscription::listFromBrightcove($api_client);

        $subscription_found = FALSE;
        foreach ($subscriptions as $subscription) {
          if ($brightcove_subscription->getEndpoint() === $subscription->getEndpoint()) {
            $subscription_found = TRUE;

            // If the endpoint exist but their ID is different, fix it.
            if ($brightcove_subscription->getBcSid() !== ($id = $subscription->getId())) {
              $brightcove_subscription->setBcSid($id);
              $brightcove_subscription->save();
            }
            break;
          }
        }

        if (!$warning_set && !$subscription_found) {
          $this->messenger->addWarning($this->t('There are subscriptions which are not available on Brightcove.<br>You can either <strong>create</strong> them on Brightcove or <strong>delete</strong> them if no longer needed.'));
          $warning_set = TRUE;
        }

        // Add create link if the subscription is missing from Brightcove.
        $create_link = [];
        if (!$subscription_found) {
          $create_link = [
            'create' => [
              'title' => $this->t('Create'),
              'url' => Url::fromRoute('entity.brightcove_subscription.create', [
                'id' => $brightcove_subscription->getId(),
              ]),
            ],
          ];
        }

        $rows[$key]['operations'] = [
          'data' => [
            '#type' => 'operations',
            '#links' => $create_link + [
              'delete' => [
                'title' => $this->t('Delete'),
                'weight' => 10,
                'url' => Url::fromRoute('entity.brightcove_subscription.delete_form', [
                  'brightcove_subscription' => $brightcove_subscription->getId(),
                ]),
              ],
            ],
          ],
        ];
      }
    }

    // Check default subscriptions for each api client.
    $api_clients_without_default_subscription = [];
    /** @var \Drupal\brightcove\Entity\ApiClient $api_client */
    foreach ($this->entityTypeManager->getStorage('brightcove_api_client')->loadMultiple() as $api_client) {
      if (Subscription::loadDefault($api_client) === NULL) {
        $api_clients_without_default_subscription[] = $api_client->getLabel();
      }
    }
    if (!empty($api_clients_without_default_subscription)) {
      $this->messenger->addWarning($this->t('There are missing default subscription(s) for the following API Client(s): %api_clients<br><a href="@link">Create missing subscription(s)</a>.', [
        '%api_clients' => implode(', ', $api_clients_without_default_subscription),
        '@link' => Url::fromRoute('entity.brightcove_subscription.create_defaults')->toString(),
      ]));
    }

    $page['subscriptions'] = [
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('There are no subscriptions yet.'),
    ];

    return $page;
  }

  /**
   * Create a subscription on Brightcove from an already existing entity.
   *
   * @param int $id
   *   BrightcoveSubscription entity ID.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to redirect user after creating a Drupal only
   *   subscription.
   */
  public function createSubscription(int $id): Response {
    try {
      $brightcove_subscription = Subscription::load($id);
      $brightcove_subscription->saveToBrightcove();
    }
    catch (SubscriptionException $e) {
      $this->messenger->addError($this->t('Failed to create Subscription on Brightcove: @error', ['@error' => $e->getMessage()]));
    }

    return $this->redirect('entity.brightcove_subscription.list');
  }

  /**
   * Enables and creates the default Subscription from Brightcove.
   *
   * @param int $id
   *   The ID of the Brightcove Subscription.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to redirect user after enabling the default
   *   subscription.
   */
  public function enable(int $id): Response {
    try {
      $subscription = Subscription::load($id);
      $subscription->saveToBrightcove();
      $this->messenger->addStatus($this->t('Default subscription for the "@api_client" API client has been successfully enabled.', ['@api_client' => $subscription->getApiClient()->label()]));
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Failed to enable the default subscription: @error', ['@error' => $e->getMessage()]));
    }
    return $this->redirect('entity.brightcove_subscription.list');
  }

  /**
   * Disabled and removed the default Subscription from Brightcove.
   *
   * @param int $id
   *   The ID of the Brightcove Subscription.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to redirect user after enabling the default
   *   subscription.
   */
  public function disable(int $id): Response {
    try {
      $subscription = Subscription::load($id);
      $subscription->deleteFromBrightcove();
      $this->messenger->addStatus($this->t('Default subscription for the "@api_client" API client has been successfully disabled.', ['@api_client' => $subscription->getApiClient()->label()]));
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Failed to disable the default subscription: @error', ['@error' => $e->getMessage()]));
    }
    return $this->redirect('entity.brightcove_subscription.list');
  }

  /**
   * Creates default subscriptions.
   *
   * This method must be called through the site's URL, otherwise the default
   * subscriptions won't be possible to create, because of the missing site URL.
   */
  public function createDefaults(): Response {
    try {
      // Get all available API clients.
      /** @var \Drupal\brightcove\Entity\ApiClientInterface[] $api_clients */
      $api_clients = $this->entityTypeManager->getStorage('brightcove_api_client')->loadMultiple();

      foreach ($api_clients as $api_client) {
        $brightcove_subscription = Subscription::loadDefault($api_client);

        // Try to grab an existing subscription by the site's endpoint URL if
        // the default doesn't exist for the current API client.
        $default_endpoint = BrightcoveUtil::getDefaultSubscriptionUrl();
        if (empty($brightcove_subscription)) {
          $brightcove_subscription = Subscription::loadByEndpoint($default_endpoint);
        }

        // If there is an existing subscription with an endpoint, make it
        // default.
        if (!empty($brightcove_subscription)) {
          $this->connection->update('brightcove_subscription')
            ->fields([
              'is_default' => 1,
            ])
            ->condition('id', $brightcove_subscription->getId())
            ->execute();
        }
        // Otherwise create a new local subscription with the site's URL.
        else {
          // Check Brightcove whether if it has a subscription for the default
          // one.
          $subscriptions = Subscription::listFromBrightcove($api_client);
          $subscription_with_default_endpoint = NULL;
          foreach ($subscriptions as $subscription) {
            if ($subscription->getEndpoint() === $default_endpoint) {
              $subscription_with_default_endpoint = $subscription;
              break;
            }
          }

          // Create a new default subscription for the API client.
          $brightcove_subscription = new Subscription(TRUE);
          $brightcove_subscription->setEvents(['video-change']);
          $brightcove_subscription->setEndpoint($default_endpoint);
          $brightcove_subscription->setApiClient($api_client);

          if ($subscription_with_default_endpoint !== NULL) {
            $brightcove_subscription->setBcSid($subscription_with_default_endpoint->getId());
            $brightcove_subscription->setStatus(TRUE);
          }
          else {
            $brightcove_subscription->setStatus(FALSE);
          }

          $brightcove_subscription->save();
        }
      }

      $this->messenger->addStatus($this->t('Default subscriptions has been successfully created.'));
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Failed to create default subscription(s), @error', ['@error' => $e->getMessage()]));
      $this->logger->logException($e, 'Failed to create default subscription(s).');
    }

    return $this->redirect('entity.brightcove_subscription.list');
  }

}
