<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Controller;

use Brightcove\Item\Video\Image;
use Brightcove\Item\Video\Link;
use Brightcove\Item\Video\Schedule;
use Brightcove\Item\Video\TextTrack;
use Brightcove\Item\Video\Video;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\Services\VideoHelperInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Video create/update API callback handler.
 */
final readonly class VideoCreateUpdateController implements ContainerInjectionInterface {

  public function __construct(
    private VideoHelperInterface $videoHelper,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('brightcove.video_helper'),
    );
  }

  /**
   * Controller callback.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request.
   * @param \Drupal\brightcove\Entity\ApiClientInterface $brightcove_api_client
   *   API client.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   */
  public function __invoke(Request $request, ApiClientInterface $brightcove_api_client): Response {
    $brightcove_videos = unserialize(base64_decode($request->getContent()), [
      'allowed_classes' => [
        Video::class,
        TextTrack::class,
        Image::class,
        Schedule::class,
        Link::class,
      ],
    ]);

    foreach ($brightcove_videos as $brightcove_video) {
      $video = $this->videoHelper->createOrUpdate($brightcove_video, $brightcove_api_client->id());

      // Get existing text tracks.
      $this->videoHelper->queueTextTracks($video, $brightcove_video);
    }
    return new Response(NULL, Response::HTTP_NO_CONTENT);
  }

}
