<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Entity;

use Drupal\brightcove\ApiClientListBuilder;
use Drupal\brightcove\Form\ApiClientDeleteForm;
use Drupal\brightcove\Form\ApiClientForm;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines the Brightcove API Client entity.
 *
 * phpcs:disable Drupal.Commenting.Deprecated.DeprecatedWrongSeeUrlFormat
 */
#[ConfigEntityType(
  id: 'brightcove_api_client',
  label: new TranslatableMarkup('API client'),
  label_collection: new TranslatableMarkup('API clients'),
  label_singular: new TranslatableMarkup('API clients'),
  label_plural: new TranslatableMarkup('API clients'),
  config_prefix: 'brightcove_api_client',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => ApiClientListBuilder::class,
    'route_provider' => [
      'html' => DefaultHtmlRouteProvider::class,
    ],
    'form' => [
      'add' => ApiClientForm::class,
      'edit' => ApiClientForm::class,
      'delete' => ApiClientDeleteForm::class,
    ],
  ],
  links: [
    'add-form' => '/admin/config/media/brightcove/api_clients/add',
    'edit-form' => '/admin/config/media/brightcove/api_clients/{brightcove_api_client}',
    'delete-form' => '/admin/config/media/brightcove/api_clients/{brightcove_api_client}/delete',
    'collection' => '/admin/config/media/brightcove/api_clients',
  ],
  admin_permission: 'administer brightcove configuration',
  label_count: [
    'singular' => '@count API client',
    'plural' => '@count API clients',
  ],
  config_export: [
    'id',
    'uuid',
    'label',
    'client_id',
    'secret_key',
    'account_id',
    'default_player',
    'max_custom_fields',
  ],
)]
final class ApiClient extends ConfigEntityBase implements ApiClientInterface {

  /**
   * The Brightcove API Client ID (Drupal-internal).
   */
  protected ?string $id = NULL;

  /**
   * The Brightcove API Client label.
   */
  protected ?string $label = NULL;

  /**
   * The Brightcove API Client account ID.
   */
  protected ?string $account_id = NULL;

  /**
   * The Brightcove API Client ID.
   */
  protected ?string $client_id = NULL;

  /**
   * The Brightcove API Client default player.
   */
  protected ?string $default_player = NULL;

  /**
   * The Brightcove API Client secret key.
   */
  protected ?string $secret_key = NULL;

  /**
   * The loaded API client.
   */
  protected ?string $client = NULL;

  /**
   * Client connection status.
   */
  protected int $client_status = self::CLIENT_INITIALIZING;

  /**
   * Client connection status message.
   *
   * If the connection status is OK, then it's an empty string.
   */
  protected string $client_status_message = '';

  /**
   * Maximum number of Custom fields.
   */
  protected int $max_custom_fields = 0;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $values, $entity_type) {
    parent::__construct($values, $entity_type);
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel(): ?string {
    return $this->label;
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountId(): ?string {
    return $this->account_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientId(): ?string {
    return $this->client_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultPlayer(): ?string {
    return $this->default_player;
  }

  /**
   * {@inheritdoc}
   */
  public function getSecretKey(): ?string {
    return $this->secret_key;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientStatus(): int {
    return $this->client_status;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientStatusMessage(): string {
    return $this->client_status_message;
  }

  /**
   * {@inheritdoc}
   */
  public function getMaxCustomFields(): int {
    return $this->max_custom_fields;
  }

  /**
   * {@inheritdoc}
   */
  public function setLabel(string $label): ApiClientInterface {
    $this->label = $label;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setAccountId(string $account_id): ApiClientInterface {
    $this->account_id = $account_id;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setClientId(string $client_id): ApiClientInterface {
    $this->client_id = $client_id;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefaultPlayer(string $default_player): ApiClientInterface {
    $this->default_player = $default_player;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setSecretKey(string $secret_key): ApiClientInterface {
    $this->secret_key = $secret_key;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setMaxCustomFields(int $max_custom_fields): ApiClientInterface {
    $this->max_custom_fields = $max_custom_fields;
    return $this;
  }

}
