<?php

declare(strict_types=1);

namespace Drupal\brightcove\Entity;

use Drupal\brightcove\CmsEntityInterface;
use Drupal\brightcove\EntityChangedFieldsTrait;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\user\UserInterface;

/**
 * Common base class for CMS entities like Video and Playlist.
 *
 * phpcs:disable Drupal.Commenting.Deprecated.DeprecatedWrongSeeUrlFormat
 */
abstract class CmsEntity extends ContentEntityBase implements CmsEntityInterface {

  use EntityChangedTrait;
  use EntityChangedFieldsTrait;

  /**
   * {@inheritdoc}
   */
  public function getName(): ?string {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName(string $name): CmsEntityInterface {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getApiClient(): ?string {
    return $this->get('api_client')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setApiClient(string $api_client): CmsEntityInterface {
    $this->set('api_client', $api_client);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): ?string {
    return $this->get('description')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription(string $description): CmsEntityInterface {
    $this->set('description', $description);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner(): UserInterface {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account): CmsEntityInterface {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId(): ?int {
    $uid = $this->get('uid');
    return $uid !== NULL ? (int) $uid->target_id : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid): CmsEntityInterface {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): ?int {
    $created = $this->get('created');
    return !empty($created->value) ? (int) $created->value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $timestamp): CmsEntityInterface {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    $this->checkUpdatedFields($storage);
    parent::preSave($storage);
  }

  /**
   * Default value callback for 'uid' base field definition.
   *
   * @see ::baseFieldDefinitions()
   *
   * @return array
   *   An array of default values.
   */
  public static function getCurrentUserId(): array {
    return [
      \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage, array &$values): void {
    parent::preCreate($storage, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

}
