<?php

declare(strict_types=1);

namespace Drupal\brightcove\Entity;

use Brightcove\Item\Playlist as BrightcovePlaylist;
use Drupal\brightcove\Access\PlaylistAccessControlHandler;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\brightcove\Entity\Storage\PlaylistStorage;
use Drupal\brightcove\Form\BrightcoveEntityDeleteForm;
use Drupal\brightcove\Form\BrightcoveInlineForm;
use Drupal\brightcove\Form\PlaylistForm;
use Drupal\brightcove\PlaylistHtmlRouteProvider;
use Drupal\brightcove\PlaylistListBuilder;
use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\EntityViewBuilder;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\brightcove\PlaylistInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\taxonomy\Entity\Term;

/**
 * Defines the Brightcove Playlist.
 *
 * @ingroup brightcove
 *
 * phpcs:disable Drupal.Commenting.Deprecated.DeprecatedWrongSeeUrlFormat
 */
#[ContentEntityType(
  id: 'brightcove_playlist',
  label: new TranslatableMarkup('Brightcove Playlist'),
  entity_keys: [
    'id' => 'bcplid',
    'label' => 'name',
    'uuid' => 'uuid',
    'uid' => 'uid',
    'langcode' => 'langcode',
    'status' => 'status',
  ],
  handlers: [
    'storage' => PlaylistStorage::class,
    'view_builder' => EntityViewBuilder::class,
    'list_builder' => PlaylistListBuilder::class,
    'views_data' => PlaylistViewsData::class,
    'form' => [
      'default' => PlaylistForm::class,
      'add' => PlaylistForm::class,
      'edit' => PlaylistForm::class,
      'delete' => BrightcoveEntityDeleteForm::class,
    ],
    'access' => PlaylistAccessControlHandler::class,
    'route_provider' => [
      'html' => PlaylistHtmlRouteProvider::class,
    ],
    'inline_form' => BrightcoveInlineForm::class,
  ],
  links: [
    'canonical' => '/brightcove_playlist/{brightcove_playlist}',
    'add-form' => '/brightcove_playlist/add',
    'edit-form' => '/brightcove_playlist/{brightcove_playlist}/edit',
    'delete-form' => '/brightcove_playlist/{brightcove_playlist}/delete',
    'collection' => '/admin/content/brightcove_playlist',
  ],
  admin_permission: 'administer brightcove playlists',
  base_table: 'brightcove_playlist',
  field_ui_base_route: 'brightcove_playlist.settings',
  constraints: [
    'brightcove_video_by_api_client_constraint' => [],
  ]
)]
class Playlist extends VideoPlaylistCmsEntity implements PlaylistInterface {

  /**
   * Get Playlist types.
   *
   * @param int|null $type
   *   Get specific type of playlist types.
   *   Possible values are TYPE_MANUAL, TYPE_SMART.
   *
   * @return array
   *   Manual and smart playlist types.
   *
   * @throws \Exception
   *   If an invalid type was given.
   *
   * @see http://docs.brightcove.com/en/video-cloud/cms-api/references/playlist-fields-reference.html
   */
  public static function getTypes(?int $type = NULL): array {
    $manual = [
      'EXPLICIT' => t('Manual: Add videos manually'),
    ];

    $smart_label = t('Smart: Add videos automatically based on tags')->render();
    $smart = [
      $smart_label => [
        'ACTIVATED_OLDEST_TO_NEWEST' => t('Smart: Activated Date (Oldest First)'),
        'ACTIVATED_NEWEST_TO_OLDEST' => t('Smart: Activated Date (Newest First)'),
        'ALPHABETICAL' => t('Smart: Video Name (A-Z)'),
        'PLAYS_TOTAL' => t('Smart: Total Plays'),
        'PLAYS_TRAILING_WEEK' => t('Smart: Trailing Week Plays'),
        'START_DATE_OLDEST_TO_NEWEST' => t('Smart: Start Date (Oldest First)'),
        'START_DATE_NEWEST_TO_OLDEST' => t('Smart: Start Date (Newest First)'),
      ],
    ];

    // Get specific type of playlist if set.
    if (!is_null($type)) {
      return match ($type) {
        self::TYPE_MANUAL => $manual,
        self::TYPE_SMART => reset($smart),
        default => throw new \Exception('The type must be either TYPE_MANUAL or TYPE_SMART'),
      };
    }

    return $manual + $smart;
  }

  /**
   * Implements callback_allowed_values_function().
   *
   * @param \Drupal\Core\Field\FieldStorageDefinitionInterface $definition
   *   The field storage definition.
   * @param \Drupal\Core\Entity\FieldableEntityInterface|null $entity
   *   (optional) The entity context if known, or NULL if the allowed values
   *   are being collected without the context of a specific entity.
   * @param bool &$cacheable
   *   (optional) If an $entity is provided, the $cacheable parameter should be
   *   modified by reference and set to FALSE if the set of allowed values
   *   returned was specifically adjusted for that entity and cannot not be
   *   reused for other entities. Defaults to TRUE.
   *
   * @return array
   *   The array of allowed values. Keys of the array are the raw stored values
   *   (number or text), values of the array are the display labels. If $entity
   *   is NULL, you should return the list of all the possible allowed values
   *   in any context so that other code (e.g. Views filters) can support the
   *   allowed values for all possible entities and bundles.
   *
   * @throws \Exception
   *   If an invalid type was given.
   */
  public static function typeAllowedValues(FieldStorageDefinitionInterface $definition, ?FieldableEntityInterface $entity = NULL, bool &$cacheable = TRUE): array {
    return self::getTypes();
  }

  /**
   * {@inheritdoc}
   */
  public function getType(): string {
    return $this->get('type')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setType($type): PlaylistInterface {
    return $this->set('type', $type);
  }

  /**
   * {@inheritdoc}
   */
  public function isFavorite(): bool {
    return (bool) $this->get('favorite')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getBrightcoveId(): ?string {
    return $this->get('playlist_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setBrightcoveId(string $id): PlaylistInterface {
    $this->set('playlist_id', $id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTagsSearchCondition(): string {
    return $this->get('tags_search_condition')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setTagsSearchCondition($condition): PlaylistInterface {
    return $this->set('tags_search_condition', $condition);
  }

  /**
   * {@inheritdoc}
   */
  public function getVideos(): array {
    return $this->get('videos')->getValue();
  }

  /**
   * {@inheritdoc}
   */
  public function setVideos(?array $videos): PlaylistInterface {
    $this->set('videos', $videos);
    return $this;
  }

  /**
   * Saves an entity permanently.
   *
   * @param bool $upload
   *   Whether to upload the video to Brightcove or not.
   *
   * @return int|bool
   *   SAVED_NEW or SAVED_UPDATED based on the operation performed.
   *   FALSE is returned if the save operation failed.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   In case of failures an exception is thrown.
   */
  public function save(bool $upload = FALSE): int|bool {
    // Check if it will be a new entity or an existing one being updated.
    $status = $this->id() ? SAVED_UPDATED : SAVED_NEW;

    // Make sure that preSave runs before any modification is made for the
    // entity.
    $saved = parent::save();

    if ($upload) {
      $cms = BrightcoveUtil::getCmsApi($this->getApiClient());

      // Setup playlist object and set minimum required values.
      $playlist = new BrightcovePlaylist();
      $playlist->setName($this->getName());

      // Save or update type if needed.
      if ($this->isFieldChanged('type')) {
        $playlist->setType($this->getType());

        // Unset search if the playlist is manual.
        if ($playlist->getType() === 'EXPLICIT') {
          $playlist->setSearch('');
          $this->setTags([]);
        }
        // Otherwise, if the playlist is smart, unset video references.
        else {
          $playlist->setVideoIds([]);
          $this->setVideos([]);
        }
      }

      // Save or update description if needed.
      if ($this->isFieldChanged('description')) {
        $playlist->setDescription($this->getDescription());
      }

      // Save or update reference ID if needed.
      if ($this->isFieldChanged('reference_id')) {
        $playlist->setReferenceId($this->getReferenceId());
      }

      // Save or update search if needed.
      if ($this->isFieldChanged('tags') || $this->isFieldChanged('tags_search_condition')) {
        $condition = '';
        if ($this->getTagsSearchCondition() === self::TAG_SEARCH_CONTAINS_ALL) {
          $condition = '+';
        }

        $tags = '';
        if (!empty($tag_items = $this->getTags())) {
          if (count($tag_items) === 1) {
            $this->setTagsSearchCondition(self::TAG_SEARCH_CONTAINS_ALL);
          }

          $tags .= $condition . 'tags:"';
          $first = TRUE;
          foreach ($tag_items as $tag) {
            $tag_term = Term::load($tag['target_id']);
            $tags .= ($first ? '' : '","') . $tag_term->getName();
            if ($first) {
              $first = FALSE;
            }
          }
          $tags .= '"';
        }

        $playlist->setSearch($tags);
      }

      // Save or update videos list if needed.
      if ($this->isFieldChanged('videos')) {
        $video_entities = $this->getVideos();
        $videos = [];
        foreach ($video_entities as $video) {
          $videos[] = Video::load($video['target_id'])->getBrightcoveId();
        }

        $playlist->setVideoIds($videos);
      }

      // Create or update a playlist.
      switch ($status) {
        case SAVED_NEW:
          // Create new playlist on Brightcove.
          $saved_playlist = $cms->createPlaylist($playlist);

          // Set the rest of the fields on BrightcoveVideo entity.
          $this->setBrightcoveId($saved_playlist->getId());
          $created_at = $saved_playlist->getCreatedAt();
          $this->setCreatedTime($created_at !== NULL ? strtotime($created_at) : 0);
          break;

        case SAVED_UPDATED:
          // Set playlist ID.
          $playlist->setId($this->getBrightcoveId());

          // Update playlist.
          $saved_playlist = $cms->updatePlaylist($playlist);
          break;
      }

      // Update changed time and playlist entity with the video ID.
      /* @phpstan-ignore-next-line The $saved_playlist variable can be empty. */
      if (isset($saved_playlist)) {
        $updated_at = $saved_playlist->getUpdatedAt();
        $this->setChangedTime($updated_at !== NULL ? strtotime($updated_at) : 0);

        // Save the entity again to save some new values which are only
        // available after creating/updating the playlist on Brightcove.
        // Also don't change the save state to show the correct message when
        // the entity is created or updated.
        parent::save();
      }
    }

    return $saved;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    // Set weights based on the real order of the fields.
    $weight = -30;

    /*
     * Drupal-specific fields first.
     *
     * bcplid - Brightcove Playlist ID (Drupal-internal).
     * uuid - UUID.
     * - "Playlist type" comes here, but that's a Brightcove-specific field.
     * - Title comes here, but that's the "Name" field from Brightcove.
     * langcode - Language.
     * api_client - Entity reference to BrightcoveAPIClient.
     * - Brightcove fields come here.
     * uid - Author.
     * created - Posted.
     * changed - Last modified.
     */
    $fields['bcplid'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('ID'))
      ->setDescription(t('The Drupal entity ID of the Brightcove Playlist.'))
      ->setReadOnly(TRUE);

    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(t('UUID'))
      ->setDescription(t('The Brightcove Playlist UUID.'))
      ->setReadOnly(TRUE);

    $fields['api_client'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('API Client'))
      ->setDescription(t('Brightcove API credentials (account) to use.'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'brightcove_api_client')
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'hidden',
        'region' => 'hidden',
        'label' => 'inline',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['player'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Player'))
      ->setDescription(t('Brightcove Player to be used for playback.'))
      ->setSetting('target_type', 'brightcove_player')
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'hidden',
        'region' => 'hidden',
        'label' => 'inline',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['type'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Playlist Type'))
      ->setRequired(TRUE)
      ->setDefaultValue('EXPLICIT')
      ->setSetting('allowed_values_function', [self::class, 'typeAllowedValues'])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'list_default',
        'label' => 'inline',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Playlist name'))
      ->setDescription(t('Title of the playlist.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 250,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'string',
        'label' => 'hidden',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Language code'))
      ->setDescription(t('The language code for the Brightcove Video.'))
      ->setDisplayOptions('form', [
        'type' => 'language_select',
        'weight' => ++$weight,
      ])
      ->setDisplayConfigurable('form', TRUE);

    /*
     * Additional Brightcove fields, based on
     * @see http://docs.brightcove.com/en/video-cloud/cms-api/references/cms-api/versions/v1/index.html#api-playlistGroup-Get_Playlists
     *
     * description - string - Playlist description
     * favorite - boolean - Whether playlist is in favorites list
     * playlist_id - string - The playlist id
     * name - string - The playlist name
     * reference_id - string - The playlist reference id
     * type - string - The playlist type: EXPLICIT or smart playlist type
     * videos - Entity reference/string array of video ids (EXPLICIT playlists
     *   only)
     * search - string - Search string to retrieve the videos (smart playlists
     *   only)
     */
    $fields['favorite'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Show Playlist in Sidebar'))
      ->setDescription(t('Whether playlist is in favorites list'))
      ->setDefaultValue(FALSE)
      ->setDisplayOptions('view', [
        'type' => 'string',
        'label' => 'inline',
        'weight' => ++$weight,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['playlist_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Playlist ID'))
      ->setDescription(t('Unique Playlist ID assigned by Brightcove.'))
      ->setReadOnly(TRUE)
      ->setDisplayOptions('view', [
        'type' => 'string',
        'label' => 'inline',
        'weight' => ++$weight,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['reference_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Reference ID'))
      ->addConstraint('UniqueField')
      ->setDescription(t('Value specified must be unique'))
      ->setSettings([
        'max_length' => 150,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'string',
        'label' => 'inline',
        'weight' => $weight,
      ])
      ->setDefaultValueCallback(static::class . '::getDefaultReferenceId')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['description'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Short description'))
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'basic_string',
        'label' => 'above',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->addPropertyConstraints('value', [
        'Length' => [
          'max' => 250,
        ],
      ]);

    $fields['tags_search_condition'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Tags search condition'))
      ->setRequired(TRUE)
      ->setDefaultValue(self::TAG_SEARCH_CONTAINS_ONE_OR_MORE)
      ->setSetting('allowed_values', [
        self::TAG_SEARCH_CONTAINS_ONE_OR_MORE => t('contains one or more'),
        self::TAG_SEARCH_CONTAINS_ALL => t('contains all'),
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => ++$weight,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['tags'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Tags'))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setSettings([
        'target_type' => 'taxonomy_term',
        'handler_settings' => [
          'target_bundles' => [VideoInterface::TAGS_VID => VideoInterface::TAGS_VID],
          'auto_create' => TRUE,
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => ++$weight,
        'settings' => [
          'autocomplete_type' => 'tags',
        ],
      ])
      ->setDisplayOptions('view', [
        'type' => 'entity_reference_label',
        'label' => 'above',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['videos'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Videos'))
      ->setDescription(t('Videos in the playlist.'))
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setSettings([
        'target_type' => 'brightcove_video',
        'handler' => 'views',
        'handler_settings' => [
          'view' => [
            'view_name' => 'brightcove_videos_by_api_client',
            'display_name' => 'videos_entity_reference',
            'arguments' => [],
          ],
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('view', [
        'type' => 'string',
        'label' => 'above',
        'weight' => $weight,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The username of the Brightcove Playlist author.'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getCurrentUserId')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => ++$weight,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => $weight,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the Brightcove Playlist is published.'))
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the Brightcove Playlist was created.'))
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'timestamp',
        'weight' => ++$weight,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the Brightcove Playlist was last edited.'))
      ->setTranslatable(TRUE);

    return $fields;
  }

}
