<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Entity\Storage;

use Brightcove\API\Request\SubscriptionRequest;
use Brightcove\Item\Subscription as BrightcoveSubscription;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\brightcove\Entity\ApiClient;
use Drupal\brightcove\Entity\Subscription;
use Drupal\brightcove\Entity\SubscriptionInterface;
use Drupal\brightcove\Entity\Exception\SubscriptionException;
use Drupal\brightcove\Services\LoggerInterface;
use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Core\Database\Connection;

/**
 * Storage for Subscriptions.
 */
final class SubscriptionStorage implements SubscriptionStorageInterface {

  /**
   * Static cache for subscriptions.
   */
  private ?array $subscriptionCache = NULL;

  /**
   * Initializes a subscription storage.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   Connection.
   * @param \Drupal\Core\Config\Entity\ConfigEntityStorageInterface $apiClientStorage
   *   API client storage.
   * @param \Drupal\brightcove\Services\LoggerInterface $logger
   *   Logger.
   */
  public function __construct(
    private readonly Connection $connection,
    private readonly ConfigEntityStorageInterface $apiClientStorage,
    private readonly LoggerInterface $logger,
  ) {
  }

  /**
   * Loads the entity by a given field and value.
   *
   * @param string $field
   *   The name of the field.
   * @param string|int $value
   *   The field's value that needs to be checked to get a specific
   *   subscription.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   The default Brightcove Subscription for the given API client or NULL if
   *   not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   *   If the field is not valid.
   */
  private function loadByField(string $field, string|int $value): ?SubscriptionInterface {
    $query = $this->connection->select('brightcove_subscription', 'bs')
      ->fields('bs');

    switch ($field) {
      case 'bcsid':
        $query->condition('bs.bcsid', $value);
        break;

      case 'default':
        $query->condition('bs.api_client_id', $value)
          ->condition('bs.is_default', 1);
        break;

      case 'endpoint':
        $query->condition('bs.endpoint', $value);
        break;

      case 'id':
        $query->condition('bs.id', $value);
        break;

      default:
        throw new SubscriptionException('Invalid field type.');
    }

    $result = $query->execute()
      ->fetchAssoc();

    if (empty($result)) {
      $result = [];
    }
    else {
      // Unserialize events.
      $result['events'] = unserialize($result['events'], [
        'allowed_classes' => FALSE,
      ]);
    }

    return self::createFromArray($result);
  }

  /**
   * {@inheritdoc}
   */
  public function loadDefault(ApiClientInterface $api_client): ?SubscriptionInterface {
    return $this->loadByField('default', $api_client->id());
  }

  /**
   * Loads the entity by its internal Drupal ID.
   *
   * @param int $id
   *   The internal Drupal ID of the entity.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   Loaded BrightcoveSubscription entity, or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function load(int $id): ?Subscription {
    return $this->loadByField('id', $id);
  }

  /**
   * {@inheritdoc}
   */
  public function loadMultiple(array $order_by = ['is_default' => 'DESC', 'endpoint' => 'ASC']): array {
    $query = $this->connection->select('brightcove_subscription', 'bs')
      ->fields('bs');

    // Set orders.
    foreach ($order_by as $field => $direction) {
      $query->orderBy($field, $direction);
    }

    $brightcove_subscriptions = $query->execute()
      ->fetchAllAssoc('id', \PDO::FETCH_ASSOC);

    $loaded_brightcove_subscriptions = [];
    foreach ($brightcove_subscriptions as $id => $brightcove_subscription) {
      $brightcove_subscription['events'] = unserialize($brightcove_subscription['events'], [
        'allowed_classes' => FALSE,
      ]);
      $loaded_brightcove_subscriptions[$id] = $this->createFromArray($brightcove_subscription);
    }
    return $loaded_brightcove_subscriptions;
  }

  /**
   * Load Subscriptions for a given API client.
   *
   * @param \Drupal\brightcove\Entity\ApiClientInterface $api_client
   *   API client.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface[]
   *   Returns loaded Brightcove Subscription entity objects keyed by ID or an
   *   empty array if there are none.
   */
  public function loadMultipleByApiClient(ApiClientInterface $api_client): array {
    $subscriptions = $this->connection->select('brightcove_subscription', 'bs')
      ->fields('bs')
      ->condition('api_client_id', $api_client->id())
      ->execute()
      ->fetchAllAssoc('id', \PDO::FETCH_ASSOC);

    $loaded_brightcove_subscriptions = [];
    foreach ($subscriptions as $id => $subscription) {
      $subscription['events'] = unserialize($subscription['events'], [
        'allowed_classes' => FALSE,
      ]);
      $loaded_brightcove_subscriptions[$id] = $this->createFromArray($subscription);
    }
    return $loaded_brightcove_subscriptions;
  }

  /**
   * Loads entity by its Brightcove Subscription ID.
   *
   * @param string $bcsid
   *   Brightcove ID of the subscription.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   Loaded BrightcoveSubscription entity, or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function loadByBcSid(string $bcsid): ?SubscriptionInterface {
    return $this->loadByField('bcsid', $bcsid);
  }

  /**
   * {@inheritdoc}
   */
  public function loadByEndpoint(string $endpoint): ?SubscriptionInterface {
    return $this->loadByField('endpoint', $endpoint);
  }

  /**
   * {@inheritdoc}
   */
  public function createFromArray(array $data): ?SubscriptionInterface {
    if ($data !== [] && !empty($data['api_client_id'])) {
      $api_client = $this->apiClientStorage->load($data['api_client_id']);
      $subscription = (new Subscription((bool) ($data['is_default'] ?? FALSE)))
        ->setApiClient($api_client)
        ->setEndpoint($data['endpoint'])
        ->setEvents($data['events']);

      if (isset($data['bcsid'])) {
        $subscription->setBcSid($data['bcsid']);
      }

      if (isset($data['status'])) {
        $subscription->setStatus((bool) $data['status']);
      }

      if (isset($data['id'])) {
        $subscription->setId((int) $data['id']);
      }

      return $subscription;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function save(SubscriptionInterface $subscription, bool $upload = FALSE): void {
    $api_client = $subscription->getApiClient();

    // Fields to insert or update.
    $fields = [
      'api_client_id' => $api_client->id(),
      'endpoint' => $subscription->getEndpoint(),
      'events' => serialize($subscription->getEvents()),
      'bcsid' => $subscription->getBcSid(),
      'status' => $subscription->isDefault() ? (int) $subscription->isActive() : 1,
    ];

    // Save new entity.
    if ($subscription->isNew()) {
      // Try to get a default subscription.
      $default_subscription = $this->loadDefault($api_client);
      $default_endpoint = BrightcoveUtil::getDefaultSubscriptionUrl();

      // Check whether we already have a default subscription for the API client
      // and throw an exception if one already exists.
      if ($subscription->isDefault() && !empty($default_subscription)) {
        throw new SubscriptionException(strtr('Default subscription already exists for the :api_client API Client.', [
          ':api_client' => $api_client->getLabel(),
        ]));
      }
      // Otherwise, if the API Client does not have a default subscription and
      // the site's URL matches the subscription's endpoint that needs to be
      // created, then make it default.
      elseif (empty($default_subscription) && $subscription->getEndpoint() === $default_endpoint) {
        $subscription->setDefault(TRUE);
      }

      // Create subscription on Brightcove only if the entity is new, as for now
      // it is not possible to update existing subscriptions.
      if ($upload) {
        $this->saveToBrightcove($subscription);
      }

      // Insert Brightcove Subscription into the database.
      $this->connection->insert('brightcove_subscription')
        ->fields($fields + ['is_default' => (int) $subscription->isDefault()])
        ->execute();
    }
    // Allow local changes to be saved.
    elseif (!$upload) {
      $this->connection->update('brightcove_subscription')
        ->fields($fields)
        ->condition('id', $subscription->getId())
        ->execute();
    }
    else {
      throw new SubscriptionException('An already existing subscription cannot be updated!');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function saveToBrightcove(SubscriptionInterface $subscription): void {
    try {
      // Get CMS API.
      $cms = BrightcoveUtil::getCmsApi($subscription->getApiClient()->id());

      if ($is_default = $subscription->isDefault()) {
        // Make sure that when the default is enabled, always use the correct
        // URL.
        $default_endpoint = BrightcoveUtil::getDefaultSubscriptionUrl();
        if ($subscription->getEndpoint() !== $default_endpoint) {
          $subscription->setEndpoint($default_endpoint);
        }
      }

      // Create subscription.
      $subscription_request = new SubscriptionRequest();
      $subscription_request->setEndpoint($subscription->getEndpoint());
      $subscription_request->setEvents($subscription->getEvents());
      $new_subscription = $cms->createSubscription($subscription_request);
      $subscription->setBcSid($new_subscription->getId());

      // If it's a default subscription update the local entity to enable it.
      if ($is_default) {
        $subscription->setStatus(TRUE);
        $this->save($subscription);
      }
    }
    catch (\Exception $e) {
      $this->logger->logException($e, 'Failed to create Subscription on Brightcove.');
      throw new SubscriptionException($e->getMessage(), $e->getCode(), $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function delete(SubscriptionInterface $subscription, bool $local_only = TRUE): void {
    $this->connection->delete('brightcove_subscription')
      ->condition('id', $subscription->getId())
      ->execute();

    if (!$local_only) {
      $this->deleteFromBrightcove($subscription);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteFromBrightcove(SubscriptionInterface $subscription): void {
    try {
      $cms = BrightcoveUtil::getCmsApi($subscription->getApiClient()->id());
      $cms->deleteSubscription($subscription->getBcSid());
    }
    catch (\Exception $e) {
      // In case of the subscription cannot be found on Brightcove, just ignore,
      // otherwise throw an exception.
      if ($e->getCode() !== 404) {
        $message = 'Failed to delete Subscription with @endpoint endpoint (ID: @bcsid).';
        $replacement = [
          '@endpoint' => $subscription->getEndpoint(),
          '@bcsid' => $subscription->getBcSid(),
        ];

        $this->logger->logException($e, $message, $replacement);
        throw new SubscriptionException(strtr($message, $replacement), $e->getCode(), $e);
      }
    }

    // In case of a default subscription set status to disabled and unset the
    // Brightcove ID.
    if ($subscription->isDefault()) {
      $subscription->setBcSid(NULL);
      $subscription->setStatus(FALSE);
      $this->save($subscription);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createOrUpdate(BrightcoveSubscription $subscription, ?ApiClient $api_client = NULL): void {
    $brightcove_subscription = $this->loadByEndpoint($subscription->getEndpoint());

    // If there is no Subscription by the endpoint, try to get one by its ID.
    if (empty($brightcove_subscription)) {
      $brightcove_subscription = $this->loadByBcSid($subscription->getId());
    }

    // Create new subscription if needed.
    if (empty($brightcove_subscription)) {
      $brightcove_subscription = new Subscription();
      $brightcove_subscription->setBcSid($subscription->getId());

      if (!empty($api_client)) {
        $brightcove_subscription->setApiClient($api_client);
      }
      else {
        return;
      }
    }

    $needs_save = FALSE;

    // Update ID.
    $bcsid = $subscription->getId();
    if ($bcsid !== $brightcove_subscription->getBcSid()) {
      $brightcove_subscription->setBcSid($bcsid);
      $needs_save = TRUE;
    }

    // In case of an inactive default subscription set status to TRUE.
    if ($brightcove_subscription->isDefault() && !$brightcove_subscription->isActive()) {
      $brightcove_subscription->setStatus(TRUE);
      $needs_save = TRUE;
    }

    // Update endpoint.
    $endpoint = $subscription->getEndpoint();
    if ($endpoint !== $brightcove_subscription->getEndpoint()) {
      $brightcove_subscription->setEndpoint($endpoint);
      $needs_save = TRUE;
    }

    // Update events.
    $events = $subscription->getEvents();
    if (!is_array($events)) {
      $events = [$events];
    }
    if ($events !== $brightcove_subscription->getEvents()) {
      $brightcove_subscription->setEvents($events);
      $needs_save = TRUE;
    }

    // Save the Subscription if needed.
    if ($needs_save) {
      $this->save($brightcove_subscription);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function count(): int {
    return (int) $this->connection->select('brightcove_subscription', 'bs')
      ->fields('bs')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * {@inheritdoc}
   */
  public function listFromBrightcove(ApiClientInterface $api_client): array {
    $subscriptions = $this->subscriptionCache;
    if ($subscriptions === NULL) {
      $cms = BrightcoveUtil::getCmsApi($api_client->id());
      $subscriptions = $cms->getSubscriptions();
    }
    return $subscriptions;
  }

}
