<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Entity\Storage;

use Brightcove\Item\Subscription as BrightcoveSubscription;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\Entity\ApiClient;
use Drupal\brightcove\Entity\Subscription;
use Drupal\brightcove\Entity\SubscriptionInterface;

/**
 * Interface for a subscription storage.
 */
interface SubscriptionStorageInterface {

  /**
   * Creates a Subscription entity from an array.
   *
   * @param array $data
   *   Array that contains information about the entity.
   *   Values:
   *     - id (int): Internal Drupal identifier, it will be ignored when saving
   *                 the entity.
   *     - bcsid (string): Brightcove Subscription entity identifier.
   *     - api_client_id (string): API Client ID.
   *     - endpoint (string): Endpoint callback URL, required.
   *     - events (string[]): Events list, e.g.: video-change, required.
   *     - is_default (bool): Whether the current Brightcove Subscription is
   *                          default or not. Will be ignored for local entity
   *                          update.
   *     - status (bool): Indicates whether a subscription is enabled or
   *                      disabled. An existing non-default subscription is
   *                      always enabled, only default subscriptions can be set
   *                      disabled.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   The initialized BrightcoveSubscription entity object, or null if the
   *   $data array is empty.
   */
  public function createFromArray(array $data): ?SubscriptionInterface;

  /**
   * Create or update a Subscription entity.
   *
   * @param \Brightcove\Item\Subscription $subscription
   *   Subscription object from Brightcove.
   * @param \Drupal\brightcove\Entity\ApiClient|null $api_client
   *   Loaded API client entity, or null.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   * @throws \Exception
   */
  public function createOrUpdate(BrightcoveSubscription $subscription, ?ApiClient $api_client = NULL): void;

  /**
   * Counts local subscriptions.
   *
   * @return int
   *   Number of the available local subscriptions entities.
   */
  public function count(): int;

  /**
   * Deletes the Brightcove Subscription.
   *
   * @param \Drupal\brightcove\Entity\SubscriptionInterface $subscription
   *   Subscription.
   * @param bool $local_only
   *   If TRUE delete the local Subscription entity only, otherwise delete the
   *   subscription from Brightcove as well.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   *   If the subscription failed to be deleted.
   */
  public function delete(SubscriptionInterface $subscription, bool $local_only = TRUE): void;

  /**
   * Delete the Subscription from Brightcove only.
   *
   * @param \Drupal\brightcove\Entity\SubscriptionInterface $subscription
   *   Subscription.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   * @throws \Exception
   */
  public function deleteFromBrightcove(SubscriptionInterface $subscription): void;

  /**
   * Get all available subscriptions from Brightcove.
   *
   * @param \Drupal\brightcove\Entity\ApiClientInterface $api_client
   *   API Client entity.
   *
   * @return \Brightcove\Item\Subscription[]
   *   List of subscriptions or null of there are none.
   */
  public function listFromBrightcove(ApiClientInterface $api_client): array;

  /**
   * Loads entity by its Brightcove Subscription ID.
   *
   * @param string $bcsid
   *   Brightcove ID of the subscription.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   Loaded BrightcoveSubscription entity, or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function loadByBcSid(string $bcsid): ?SubscriptionInterface;

  /**
   * Load a Subscription by its endpoint.
   *
   * @param string $endpoint
   *   The endpoint.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   The Subscription with the given endpoint or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function loadByEndpoint(string $endpoint): ?SubscriptionInterface;

  /**
   * Loads the entity by its internal Drupal ID.
   *
   * @param int $id
   *   The internal Drupal ID of the entity.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   Loaded BrightcoveSubscription entity, or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function load(int $id): ?Subscription;

  /**
   * Loads the default subscription by API Client ID.
   *
   * @param \Drupal\brightcove\Entity\ApiClientInterface $api_client
   *   Loaded API Client entity.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   The default Brightcove Subscription for the given API client or NULL if
   *   not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public function loadDefault(ApiClientInterface $api_client): ?SubscriptionInterface;

  /**
   * Loads multiple BrightcoveSubscription entities.
   *
   * @param string[] $order_by
   *   Fields to order by:
   *     - key: the name of the field.
   *     - value: the order direction.
   *
   * @return \Drupal\brightcove\Entity\Subscription[]
   *   Returns loaded Brightcove Subscription entity objects keyed by ID or an
   *   empty array if there are none.
   */
  public function loadMultiple(array $order_by = ['is_default' => 'DESC', 'endpoint' => 'ASC']): array;

  /**
   * Load Subscriptions for a given API client.
   *
   * @param \Drupal\brightcove\Entity\ApiClient $api_client
   *   Loaded API client.
   *
   * @return \Drupal\brightcove\Entity\Subscription[]
   *   Returns loaded Brightcove Subscription entity objects keyed by ID or an
   *   empty array if there are none.
   */
  public function loadMultipleByApiClient(ApiClient $api_client): array;

  /**
   * Saves the subscription entity.
   *
   * @param \Drupal\brightcove\Entity\SubscriptionInterface $subscription
   *   Subscription entity.
   * @param bool $upload
   *   Whether to upload the new subscription to Brightcove or not.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   *   If an error happens during saving the subscription.
   */
  public function save(SubscriptionInterface $subscription, bool $upload = FALSE): void;

  /**
   * Saves the subscription entity to Brightcove.
   *
   * @param \Drupal\brightcove\Entity\SubscriptionInterface $subscription
   *   Subscription.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   *   If the Subscription wasn't saved to Brightcove successfully.
   */
  public function saveToBrightcove(SubscriptionInterface $subscription): void;

}
