<?php

declare(strict_types=1);

namespace Drupal\brightcove\Entity;

use Brightcove\API\Request\SubscriptionRequest;
use Brightcove\Item\Subscription as BrightcoveSubscription;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\brightcove\Entity\Exception\SubscriptionException;
use Drupal\Core\Database\Connection;

/**
 * Defines the Subscription entity.
 */
class Subscription implements SubscriptionInterface {

  /**
   * Static cache for subscriptions.
   */
  protected static ?BrightcoveSubscription $subscriptionCache = NULL;

  /**
   * Internal Drupal ID.
   */
  private ?int $id = NULL;

  /**
   * Brightcove Subscription ID of the entity.
   */
  private ?string $bcsid = NULL;

  /**
   * Status of the Subscription.
   */
  private bool $status = TRUE;

  /**
   * The Brightcove API Client.
   */
  private ?ApiClientInterface $apiClient = NULL;

  /**
   * The notifications endpoint.
   */
  private ?string $endpoint = NULL;

  /**
   * Array of events subscribed to.
   */
  private array $events = [];

  /**
   * Drupal database connection.
   */
  private Connection $connection;

  /**
   * {@inheritdoc}
   */
  public function isActive(): bool {
    if ($this->default) {
      return $this->status;
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isDefault(): bool {
    return $this->default;
  }

  /**
   * {@inheritdoc}
   */
  public function isNew(): bool {
    return empty($this->id);
  }

  /**
   * {@inheritdoc}
   */
  public function getApiClient(): ?ApiClientInterface {
    return !empty($this->apiClient) ? $this->apiClient : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getBcSid(): ?string {
    return $this->bcsid;
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoint(): ?string {
    return $this->endpoint;
  }

  /**
   * {@inheritdoc}
   */
  public function getEvents(): array {
    return $this->events;
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): ?int {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  public function setApiClient(?ApiClientInterface $api_client): SubscriptionInterface {
    $this->apiClient = $api_client;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setBcSid(?string $bcsid): SubscriptionInterface {
    $this->bcsid = $bcsid;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefault(bool $is_default): SubscriptionInterface {
    $this->default = $is_default;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setEndpoint(string $endpoint): SubscriptionInterface {
    $this->endpoint = $endpoint;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setEvents(array $events): SubscriptionInterface {
    $this->events = $events;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setId(int $id): SubscriptionInterface {
    $this->id = $id;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(bool $status): SubscriptionInterface {
    if ($this->default) {
      $this->status = $status;
      return $this;
    }
    throw new \Exception('Not possible to set status of a non-default Subscription.');
  }

  /**
   * Initializes the BrightcoveSubscription Entity object.
   *
   * @param bool $default
   *   Whether this subscription should be default or not. There is being only
   *   one per API client.
   */
  public function __construct(
    private bool $default = FALSE,
  ) {
    $this->id = NULL;
    /* @phpstan-ignore globalDrupalDependencyInjection.useDependencyInjection */
    $this->connection = \Drupal::getContainer()->get('database');
  }

  /**
   * Loads the entity by a given field and value.
   *
   * @param string $field
   *   The name of the field.
   * @param string|int $value
   *   The field's value that needs to be checked to get a specific
   *   subscription.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   The default Brightcove Subscription for the given API client or NULL if
   *   not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   *   If the field is not valid.
   */
  protected static function loadByField(string $field, $value): ?SubscriptionInterface {
    /** @var \Drupal\Core\Database\Connection $connection */
    $connection = \Drupal::getContainer()
      ->get('database');

    $query = $connection->select('brightcove_subscription', 'bs')
      ->fields('bs');

    switch ($field) {
      case 'bcsid':
        $query->condition('bs.bcsid', $value);
        break;

      case 'default':
        $query->condition('bs.api_client_id', $value)
          ->condition('bs.is_default', 1);
        break;

      case 'endpoint':
        $query->condition('bs.endpoint', $value);
        break;

      case 'id':
        $query->condition('bs.id', $value);
        break;

      default:
        throw new SubscriptionException('Invalid field type.');
    }

    $result = $query->execute()
      ->fetchAssoc();

    if (empty($result)) {
      $result = [];
    }
    else {
      // Unserialize events.
      $result['events'] = unserialize($result['events'], [
        'allowed_classes' => FALSE,
      ]);
    }

    return self::createFromArray($result);
  }

  /**
   * Loads the default subscription by API Client ID.
   *
   * @param \Drupal\brightcove\Entity\ApiClient $api_client
   *   Loaded API Client entity.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   The default Brightcove Subscription for the given API client or NULL if
   *   not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public static function loadDefault(ApiClient $api_client): ?Subscription {
    return self::loadByField('default', $api_client->id());
  }

  /**
   * Loads the entity by its internal Drupal ID.
   *
   * @param int $id
   *   The internal Drupal ID of the entity.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   Loaded BrightcoveSubscription entity, or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public static function load(int $id): ?Subscription {
    return self::loadByField('id', $id);
  }

  /**
   * Loads multiple BrightcoveSubscription entities.
   *
   * @param string[] $order_by
   *   Fields to order by:
   *     - key: the name of the field.
   *     - value: the order direction.
   *
   * @return \Drupal\brightcove\Entity\Subscription[]
   *   Returns loaded Brightcove Subscription entity objects keyed by ID or an
   *   empty array if there are none.
   */
  public static function loadMultiple(array $order_by = ['is_default' => 'DESC', 'endpoint' => 'ASC']): array {
    /** @var \Drupal\Core\Database\Connection $connection */
    $connection = \Drupal::getContainer()
      ->get('database');

    $query = $connection->select('brightcove_subscription', 'bs')
      ->fields('bs');

    // Set orders.
    foreach ($order_by as $field => $direction) {
      $query->orderBy($field, $direction);
    }

    $brightcove_subscriptions = $query->execute()
      ->fetchAllAssoc('id', \PDO::FETCH_ASSOC);

    $loaded_brightcove_subscriptions = [];
    foreach ($brightcove_subscriptions as $id => $brightcove_subscription) {
      $brightcove_subscription['events'] = unserialize($brightcove_subscription['events'], [
        'allowed_classes' => FALSE,
      ]);
      $loaded_brightcove_subscriptions[$id] = Subscription::createFromArray($brightcove_subscription);
    }
    return $loaded_brightcove_subscriptions;
  }

  /**
   * Load Subscriptions for a given API client.
   *
   * @param \Drupal\brightcove\Entity\ApiClient $api_client
   *   Loaded API client.
   *
   * @return \Drupal\brightcove\Entity\Subscription[]
   *   Returns loaded Brightcove Subscription entity objects keyed by ID or an
   *   empty array if there are none.
   */
  public static function loadMultipleByApiClient(ApiClient $api_client): array {
    /** @var \Drupal\Core\Database\Connection $connection */
    $connection = \Drupal::getContainer()
      ->get('database');

    $brightcove_subscriptions = $connection->select('brightcove_subscription', 'bs')
      ->fields('bs')
      ->condition('api_client_id', $api_client->id())
      ->execute()
      ->fetchAllAssoc('id', \PDO::FETCH_ASSOC);

    $loaded_brightcove_subscriptions = [];
    foreach ($brightcove_subscriptions as $id => $brightcove_subscription) {
      $brightcove_subscription['events'] = unserialize($brightcove_subscription['events'], [
        'allowed_classes' => FALSE,
      ]);
      $loaded_brightcove_subscriptions[$id] = Subscription::createFromArray($brightcove_subscription);
    }
    return $loaded_brightcove_subscriptions;
  }

  /**
   * Loads entity by its Brightcove Subscription ID.
   *
   * @param string $bcsid
   *   Brightcove ID of the subscription.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   Loaded BrightcoveSubscription entity, or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public static function loadByBcSid(string $bcsid): ?SubscriptionInterface {
    return self::loadByField('bcsid', $bcsid);
  }

  /**
   * Load a Subscription by its endpoint.
   *
   * @param string $endpoint
   *   The endpoint.
   *
   * @return \Drupal\brightcove\Entity\SubscriptionInterface|null
   *   The Subscription with the given endpoint or NULL if not found.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   */
  public static function loadByEndpoint(string $endpoint): ?SubscriptionInterface {
    return self::loadByField('endpoint', $endpoint);
  }

  /**
   * Creates a BrightcoveSubscription entity from an array.
   *
   * @param array $data
   *   Array that contains information about the entity.
   *   Values:
   *     - id (int): Internal Drupal identifier, it will be ignored when saving
   *                 the entity.
   *     - bcsid (string): Brightcove Subscription entity identifier.
   *     - api_client_id (string): API Client ID.
   *     - endpoint (string): Endpoint callback URL, required.
   *     - events (string[]): Events list, eg.: video-change, required.
   *     - is_default (bool): Whether the current Brightcove Subscription is
   *                          default or not. Will be ignored for local entity
   *                          update.
   *     - status (bool): Indicates whether a subscription is enabled or
   *                      disabled. An existing non-default subscription is
   *                      always enabled, only default subscriptions can be set
   *                      to disabled.
   *
   * @return \Drupal\brightcove\Entity\Subscription|null
   *   The initialized BrightcoveSubscription entity object, or null if the
   *   $data array is empty.
   */
  public static function createFromArray(array $data): ?Subscription {
    if (!empty($data) && !empty($data['api_client_id'])) {
      $api_client = ApiClient::load($data['api_client_id']);
      $brightcove_subscription = (new Subscription())
        ->setApiClient($api_client)
        ->setEndpoint($data['endpoint'])
        ->setEvents($data['events']);

      if (isset($data['id'])) {
        $brightcove_subscription->id = (int) $data['id'];
      }
      if (isset($data['bcsid'])) {
        $brightcove_subscription->bcsid = $data['bcsid'];
      }
      if (isset($data['is_default'])) {
        $brightcove_subscription->default = (bool) $data['is_default'];
      }
      if (isset($data['status'])) {
        $brightcove_subscription->status = (bool) $data['status'];
      }

      return $brightcove_subscription;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function save(bool $upload = FALSE): void {
    // Fields to insert or update.
    $fields = [
      'api_client_id' => $this->getApiClient()->id(),
      'endpoint' => $this->getEndpoint(),
      'events' => serialize($this->getEvents()),
    ];
    $fields += ['bcsid' => !empty($this->bcsid) ? $this->getBcSid() : NULL];
    $fields += ['status' => $this->isDefault() ? (int) $this->isActive() : 1];

    // Save new entity.
    if ($this->isNew()) {
      // Try to get a default subscription.
      $default_subscription = self::loadDefault($this->apiClient);
      $default_endpoint = BrightcoveUtil::getDefaultSubscriptionUrl();

      // Check whether we already have a default subscription for the API client
      // and throw an exception if one already exists.
      if ($this->isDefault() && !empty($default_subscription)) {
        throw new SubscriptionException(strtr('Default subscription already exists for the :api_client API Client.', [
          ':api_client' => $this->apiClient->getLabel(),
        ]));
      }
      // Otherwise, if the API Client does not have a default subscription and
      // the site's URL matches the subscription's endpoint that needs to be
      // created, then make it default.
      elseif (empty($default_subscription) && $this->getEndpoint() === $default_endpoint) {
        $this->default = TRUE;
      }

      // Create subscription on Brightcove only if the entity is new, as for now
      // it is not possible to update existing subscriptions.
      if ($upload) {
        $this->saveToBrightcove();
      }

      // Insert Brightcove Subscription into the database.
      $this->connection->insert('brightcove_subscription')
        ->fields($fields + ['is_default' => (int) $this->isDefault()])
        ->execute();
    }
    // Allow local changes to be saved.
    elseif (!$upload) {
      $this->connection->update('brightcove_subscription')
        ->fields($fields)
        ->condition('id', $this->getId())
        ->execute();
    }
    else {
      throw new SubscriptionException('An already existing subscription cannot be updated!');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function saveToBrightcove(): void {
    try {
      // Get CMS API.
      $cms = BrightcoveUtil::getCmsApi($this->apiClient->id());

      if ($is_default = $this->isDefault()) {
        // Make sure that when the default is enabled, always use the correct
        // URL.
        $default_endpoint = BrightcoveUtil::getDefaultSubscriptionUrl();
        if ($this->endpoint !== $default_endpoint) {
          $this->setEndpoint($default_endpoint);
        }
      }

      // Create subscription.
      $subscription_request = new SubscriptionRequest();
      $subscription_request->setEndpoint($this->getEndpoint());
      $subscription_request->setEvents($this->getEvents());
      $new_subscription = $cms->createSubscription($subscription_request);
      $this->setBcSid($new_subscription->getId());

      // If it's a default subscription update the local entity to enable it.
      if ($is_default) {
        $this->setStatus(TRUE);
        $this->save();
      }
    }
    catch (\Exception $e) {
      /* @phpstan-ignore globalDrupalDependencyInjection.useDependencyInjection */
      \Drupal::getContainer()->get('brightcove.logger')->logException($e, 'Failed to create Subscription on Brightcove.');
      throw new SubscriptionException($e->getMessage(), $e->getCode(), $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function delete(bool $local_only = TRUE): void {
    $this->connection->delete('brightcove_subscription')
      ->condition('id', $this->id)
      ->execute();

    if (!$local_only) {
      $this->deleteFromBrightcove();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteFromBrightcove(): void {
    try {
      $cms = BrightcoveUtil::getCmsApi($this->apiClient->id());
      $cms->deleteSubscription($this->getBcSid());
    }
    catch (\Exception $e) {
      // In case of the subscription cannot be found on Brightcove, just ignore,
      // otherwise throw an exception.
      if ($e->getCode() !== 404) {
        $message = 'Failed to delete Subscription with @endpoint endpoint (ID: @bcsid).';
        $replacement = [
          '@endpoint' => $this->getEndpoint(),
          '@bcsid' => $this->getBcSid(),
        ];

        /* @phpstan-ignore globalDrupalDependencyInjection.useDependencyInjection */
        \Drupal::getContainer()->get('brightcove.logger')->logException($e, $message, $replacement);
        throw new SubscriptionException(strtr($message, $replacement), $e->getCode(), $e);
      }
    }

    // In case of a default subscription set status to disabled and unset the
    // Brightcove ID.
    if ($this->isDefault()) {
      $this->setBcSid(NULL);
      $this->setStatus(FALSE);
      $this->save();
    }
  }

  /**
   * Create or update a Subscription entity.
   *
   * @param \Brightcove\Item\Subscription $subscription
   *   Subscription object from Brightcove.
   * @param \Drupal\brightcove\Entity\ApiClientInterface|null $api_client
   *   Loaded API client entity, or null.
   *
   * @throws \Drupal\brightcove\Entity\Exception\SubscriptionException
   * @throws \Exception
   */
  public static function createOrUpdate(BrightcoveSubscription $subscription, ?ApiClientInterface $api_client = NULL): void {
    $brightcove_subscription = self::loadByEndpoint($subscription->getEndpoint());

    // If there is no Subscription by the endpoint, try to get one by its ID.
    if ($brightcove_subscription !== NULL) {
      $brightcove_subscription = self::loadByBcSid($subscription->getId());
    }
    // Create new subscription if needed.
    else {
      $brightcove_subscription = new Subscription();
      $brightcove_subscription->bcsid = $subscription->getId();

      if (!empty($api_client)) {
        $brightcove_subscription->setApiClient($api_client);
      }
      else {
        return;
      }
    }

    $needs_save = FALSE;

    // Update ID.
    $bcsid = $subscription->getId();
    if ($bcsid !== $brightcove_subscription->getBcSid()) {
      $brightcove_subscription->setBcSid($bcsid);
      $needs_save = TRUE;
    }

    // In case of an inactive default subscription set status to TRUE.
    if ($brightcove_subscription->isDefault() && !$brightcove_subscription->isActive()) {
      $brightcove_subscription->setStatus(TRUE);
      $needs_save = TRUE;
    }

    // Update endpoint.
    $endpoint = $subscription->getEndpoint();
    if ($endpoint !== $brightcove_subscription->getEndpoint()) {
      $brightcove_subscription->setEndpoint($endpoint);
      $needs_save = TRUE;
    }

    // Update events.
    $events = $subscription->getEvents();
    if (!is_array($events)) {
      $events = [$events];
    }
    if ($events !== $brightcove_subscription->getEvents()) {
      $brightcove_subscription->setEvents($events);
      $needs_save = TRUE;
    }

    // Save the Subscription if needed.
    if ($needs_save) {
      $brightcove_subscription->save();
    }
  }

  /**
   * Counts local subscriptions.
   *
   * @return int
   *   Number of the available local subscriptions entities.
   */
  public static function count(): int {
    /** @var \Drupal\Core\Database\Connection $connection */
    $connection = \Drupal::getContainer()
      ->get('database');

    return (int) $connection->select('brightcove_subscription', 'bs')
      ->fields('bs')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Get all available subscriptions from Brightcove.
   *
   * @param \Drupal\brightcove\Entity\ApiClientInterface $api_client
   *   API Client entity.
   *
   * @return \Brightcove\Item\Subscription[]
   *   List of subscriptions or null of there are none.
   */
  public static function listFromBrightcove(ApiClientInterface $api_client): array {
    $subscriptions = static::$subscriptionCache;
    if ($subscriptions === NULL) {
      $cms = BrightcoveUtil::getCmsApi($api_client->id());
      $subscriptions = $cms->getSubscriptions();
    }
    return $subscriptions;
  }

}
