<?php

declare(strict_types=1);

namespace Drupal\brightcove\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\EntityOwnerInterface;

/**
 * Provides an interface for defining Text Track entities.
 *
 * @ingroup brightcove
 */
interface TextTrackInterface extends ContentEntityInterface, EntityChangedInterface, EntityOwnerInterface {

  /**
   * Returns the name of Text Track.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|string|null
   *   The name of the Text Track.
   */
  public function getName(): TranslatableMarkup|string|null;

  /**
   * Sets the name of the Text Track.
   *
   * @param string $name
   *   The name of the Text Track.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setName(string $name): TextTrackInterface;

  /**
   * Returns the WebVTT file entity.
   *
   * @return array
   *   WebVTT file entity.
   */
  public function getWebVttFile(): array;

  /**
   * Sets the WebVTT file.
   *
   * @param array $file
   *   The WebVTT file entity.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setWebVttFile(array $file): TextTrackInterface;

  /**
   * Returns the Brightcove ID of the Text Track.
   *
   * @return string|null
   *   The Brightcove ID of the Text Track.
   */
  public function getTextTrackId(): ?string;

  /**
   * Sets the Brightcove ID of the Text Track.
   *
   * @param string $text_track_id
   *   The Brightcove ID of the Text Track.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setTextTrackId(string $text_track_id): TextTrackInterface;

  /**
   * Returns the source link.
   *
   * @return array|null
   *   The related link.
   */
  public function getSource(): ?array;

  /**
   * Sets the source link.
   *
   * @param string $source
   *   The related link.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setSource(string $source): TextTrackInterface;

  /**
   * Returns the source language.
   *
   * @return string|null
   *   The 2-letter source language.
   */
  public function getSourceLanguage(): ?string;

  /**
   * Sets the source language.
   *
   * @param string $source_language
   *   A 2-letter source language code, e.g.: hu, en.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setSourceLanguage(string $source_language): TextTrackInterface;

  /**
   * Gets the Brightcove Text Track name.
   *
   * @return string|null
   *   Name of the Brightcove Text Track.
   */
  public function getLabel(): ?string;

  /**
   * Sets the Brightcove Text Track name.
   *
   * @param string $label
   *   The Brightcove Text Track name.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called Brightcove Text Track entity.
   */
  public function setLabel(string $label): TextTrackInterface;

  /**
   * Returns the Text Tracks's kind.
   *
   * @return string|null
   *   The kind of the Text Track.
   */
  public function getKind(): ?string;

  /**
   * Sets the Text Tracks's kind.
   *
   * @param string $kind
   *   The Text Tracks's kind.
   *   Possible values are:
   *     - BrightcoveTextTrack::KIND_CAPTIONS
   *     - BrightcoveTextTrack::KIND_SUBTITLES
   *     - BrightcoveTextTrack::KIND_DESCRIPTION
   *     - BrightcoveTextTrack::KIND_CHAPTERS
   *     - BrightcoveTextTrack::KIND_METADATA
   *   .
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setKind(string $kind): TextTrackInterface;

  /**
   * Returns the mime type.
   *
   * @return string|null
   *   The mime type.
   */
  public function getMimeType(): ?string;

  /**
   * Sets the mime type.
   *
   * @param string $mime_type
   *   The mime type.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setMimeType(string $mime_type): TextTrackInterface;

  /**
   * Returns the asset ID.
   *
   * Only for managed text tracks.
   *
   * @return string|null
   *   Asset ID.
   */
  public function getAssetId(): ?string;

  /**
   * Sets the Asset ID.
   *
   * @param string $asset_id
   *   The asset ID.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setAssetId(string $asset_id): TextTrackInterface;

  /**
   * Returns a list of text track sources.
   *
   * @return array
   *   List of text track sources.
   */
  public function getSources(): array;

  /**
   * Set text track sources.
   *
   * @param array $sources
   *   Text track sources.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setSources(array $sources): TextTrackInterface;

  /**
   * Whether the text track is default or not.
   *
   * @return bool
   *   TRUE if the text track is the default one, FALSE otherwise.
   */
  public function isDefault(): bool;

  /**
   * Set Text Track as default.
   *
   * @param bool $default
   *   TRUE or FALSE whether is the Text Track is the default one or not.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called BrightcoveTextTrack entity.
   */
  public function setDefault(bool $default): TextTrackInterface;

  /**
   * Gets the Brightcove Text Track creation timestamp.
   *
   * @return int
   *   Creation timestamp of the Brightcove Text Track.
   */
  public function getCreatedTime(): int;

  /**
   * Sets the Brightcove Text Track creation timestamp.
   *
   * @param int $timestamp
   *   The Brightcove Text Track creation timestamp.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called Brightcove Text Track entity.
   */
  public function setCreatedTime(int $timestamp): TextTrackInterface;

  /**
   * Returns the Brightcove Text Track published status indicator.
   *
   * Unpublished Brightcove Text Track are only visible to restricted users.
   *
   * @return bool
   *   TRUE if the Brightcove Text Track is published.
   */
  public function isPublished(): bool;

  /**
   * Sets the published status of a Brightcove Text Track.
   *
   * @param bool $published
   *   TRUE to set this Brightcove Text Track to published, FALSE to set it to
   *   unpublished.
   *
   * @return \Drupal\brightcove\Entity\TextTrackInterface
   *   The called Brightcove Text Track entity.
   */
  public function setPublished(bool $published): TextTrackInterface;

}
