<?php

declare(strict_types=1);

namespace Drupal\brightcove\Form;

use Drupal\brightcove\Entity\Subscription;
use Drupal\brightcove\Services\QueueHandlerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityConfirmFormBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Builds the form to delete Brightcove API Client entities.
 */
class ApiClientDeleteForm extends EntityConfirmFormBase {

  use DependencySerializationTrait;

  /**
   * Initializes an API client delete form builder.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Database\Connection $connection
   *   Database connection.
   * @param \Drupal\brightcove\Services\QueueHandlerInterface $queueHandler
   *   Queue handler.
   * @param \Drupal\Core\Queue\QueueInterface $playlistLocalDeleteQueue
   *   Playlist local delete queue worker.
   * @param \Drupal\Core\Queue\QueueInterface $videoLocalDeleteQueue
   *   Video local delete queue worker.
   * @param \Drupal\Core\Queue\QueueInterface $playerDeleteQueue
   *   Player local delete queue worker.
   * @param \Drupal\Core\Queue\QueueInterface $customFieldDeleteQueue
   *   Custom field local delete queue worker.
   * @param \Drupal\Core\Queue\QueueInterface $textTrackDeleteQueue
   *   Text track delete queue object.
   * @param \Drupal\Core\Queue\QueueInterface $subscriptionDeleteQueue
   *   Subscription delete queue object.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    protected readonly Connection $connection,
    protected readonly QueueHandlerInterface $queueHandler,
    protected readonly QueueInterface $playlistLocalDeleteQueue,
    protected readonly QueueInterface $videoLocalDeleteQueue,
    protected readonly QueueInterface $playerDeleteQueue,
    protected readonly QueueInterface $customFieldDeleteQueue,
    protected readonly QueueInterface $textTrackDeleteQueue,
    protected readonly QueueInterface $subscriptionDeleteQueue,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('brightcove.queue_handler'),
      $container->get('queue')->get('brightcove_playlist_local_delete_queue_worker'),
      $container->get('queue')->get('brightcove_video_local_delete_queue_worker'),
      $container->get('queue')->get('brightcove_player_delete_queue_worker'),
      $container->get('queue')->get('brightcove_custom_field_delete_queue_worker'),
      $container->get('queue')->get('brightcove_text_track_delete_queue_worker'),
      $container->get('queue')->get('brightcove_subscription_delete_queue_worker')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete %name?', ['%name' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone.<br>Warning: By deleting API Client all of its local contents will be deleted too, including videos, playlists, players, custom fields and subscriptions.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('entity.brightcove_api_client.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\brightcove\Entity\ApiClient $entity */
    $entity = $this->entity;

    // Empty queues.
    foreach ($this->queueHandler->getStatusQueues() as $queue_name) {
      $this->queueHandler->clearQueue($queue_name);
    }

    // Collect all playlists belonging for the api client.
    $playlists = $this->entityTypeManager->getStorage('brightcove_playlist')
      ->getQuery()
      ->condition('api_client', $entity->id())
      ->accessCheck()
      ->execute();
    foreach ($playlists as $playlist) {
      $this->playlistLocalDeleteQueue->createItem($playlist);
    }

    // Collect all text tracks belonging for the api client.
    $query = $this->connection->select('brightcove_text_track', 'btt')
      ->fields('btt', ['text_track_id']);
    $query->innerJoin('brightcove_video__text_tracks', 'bvtt', '%alias.text_tracks_target_id = btt.bcttid');
    $query->innerJoin('brightcove_video', 'bv', '%alias.bcvid = bvtt.entity_id');
    $text_tracks = $query->condition('api_client', $entity->id())
      ->execute();
    foreach ($text_tracks as $text_track) {
      $this->textTrackDeleteQueue->createItem($text_track->text_track_id);
    }

    // Collect all videos belonging for the api client.
    $videos = $this->entityTypeManager->getStorage('brightcove_video')
      ->getQuery()
      ->condition('api_client', $entity->id())
      ->accessCheck()
      ->execute();
    foreach ($videos as $video) {
      $this->videoLocalDeleteQueue->createItem($video);
    }

    // Collect all players belonging for the api client.
    $players = $this->entityTypeManager->getStorage('brightcove_player')
      ->getQuery()
      ->condition('api_client', $entity->id())
      ->accessCheck()
      ->execute();
    foreach ($players as $player) {
      $this->playerDeleteQueue->createItem(['player_entity_id' => $player]);
    }

    // Collect all custom fields belonging for the api client.
    $custom_fields = $this->entityTypeManager->getStorage('brightcove_custom_field')
      ->getQuery()
      ->condition('api_client', $entity->id())
      ->accessCheck()
      ->execute();
    foreach ($custom_fields as $custom_field) {
      $this->customFieldDeleteQueue->createItem($custom_field);
    }

    // First delete the default subscription from Brightcove if it's active.
    $default_subscription = Subscription::loadDefault($entity);
    if (!empty($default_subscription)) {
      if ($default_subscription->isActive()) {
        $default_subscription->delete();
      }
      else {
        $default_subscription->delete(TRUE);
      }
    }

    // Then collect all available subscriptions belonging to the given API
    // client, and put them into the delete queue.
    $brightcove_subscriptions = Subscription::loadMultipleByApiClient($entity);
    foreach ($brightcove_subscriptions as $brightcove_subscription) {
      $this->subscriptionDeleteQueue->createItem([
        'subscription_id' => $brightcove_subscription->getBcSid(),
        'local_only' => TRUE,
      ]);
    }

    // Initialize batch.
    batch_set([
      'title' => $this->t('Brightcove sync'),
      'operations' => [
        [
          [$this->queueHandler, 'runQueue'], ['brightcove_playlist_local_delete_queue_worker'],
        ],
        [
          [$this->queueHandler, 'runQueue'], ['brightcove_video_local_delete_queue_worker'],
        ],
        [
          [$this->queueHandler, 'runQueue'], ['brightcove_player_delete_queue_worker'],
        ],
        [
          [$this->queueHandler, 'runQueue'], ['brightcove_custom_field_delete_queue_worker'],
        ],
        [
          [$this->queueHandler, 'runQueue'], ['brightcove_text_track_delete_queue_worker'],
        ],
        [
          [$this->queueHandler, 'runQueue'], ['brightcove_subscription_delete_queue_worker'],
        ],
      ],
    ]);

    // Delete api client.
    $entity->delete();
    $this->messenger()->addStatus($this->t('Entity @type: deleted @label.', [
      '@type' => $this->entity->bundle(),
      '@label' => $this->entity->label(),
    ]));

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
