<?php

declare(strict_types=1);

namespace Drupal\brightcove\Form;

use Brightcove\API\Exception\ApiException;
use Brightcove\API\Exception\AuthenticationException;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\brightcove\Entity\Subscription;
use Drupal\brightcove\Services\ApiClientHandlerInterface;
use Drupal\brightcove\Services\CmsApiHandlerInterface;
use Drupal\brightcove\Services\PmApiHandlerInterface;
use Drupal\brightcove\Services\QueueHandlerInterface;
use Drupal\brightcove\Services\SettingsInterface;
use Drupal\brightcove\Type\ApiClientStatusType;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Psr\Http\Client\ClientExceptionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form builder for API client.
 */
final class ApiClientForm extends EntityForm {

  /**
   * Brightcove API client config entity.
   *
   * @var \Drupal\brightcove\Entity\ApiClientInterface
   */
  protected $entity;

  /**
   * Initializes an API client form builder.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\brightcove\Services\SettingsInterface $settings
   *   Module specific settings.
   * @param \Drupal\brightcove\Services\QueueHandlerInterface $queueHandler
   *   Queue handler.
   * @param \Drupal\Core\Queue\QueueInterface $playerQueue
   *   Player queue.
   * @param \Drupal\Core\Queue\QueueInterface $customFieldQueue
   *   Custom field queue.
   * @param \Drupal\Core\Queue\QueueInterface $subscriptionsQueue
   *   Custom field queue.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   String translation service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger.
   * @param \Drupal\brightcove\Services\ApiClientHandlerInterface $apiClientHandler
   *   API client handler.
   * @param \Drupal\brightcove\Services\CmsApiHandlerInterface $cmsApiHandler
   *   CMS API handler.
   * @param \Drupal\brightcove\Services\PmApiHandlerInterface $pmApiHandler
   *   PM API handler.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    private readonly SettingsInterface $settings,
    private readonly QueueHandlerInterface $queueHandler,
    private readonly QueueInterface $playerQueue,
    private readonly QueueInterface $customFieldQueue,
    private readonly QueueInterface $subscriptionsQueue,
    TranslationInterface $string_translation,
    MessengerInterface $messenger,
    private readonly ApiClientHandlerInterface $apiClientHandler,
    private readonly CmsApiHandlerInterface $cmsApiHandler,
    private readonly PmApiHandlerInterface $pmApiHandler,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->stringTranslation = $string_translation;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $queue = $container->get('queue');

    return new self(
      $container->get('entity_type.manager'),
      $container->get('brightcove.settings'),
      $container->get('brightcove.queue_handler'),
      $queue->get('brightcove_player_queue_worker'),
      $queue->get('brightcove_custom_field_queue_worker'),
      $queue->get('brightcove_subscriptions_queue_worker'),
      $container->get('string_translation'),
      $container->get('messenger'),
      $container->get('brightcove.api.client'),
      $container->get('brightcove.api.cms'),
      $container->get('brightcove.api.pm'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    // Don't even try reporting the status/error message of a new client.
    if (!$this->entity->isNew()) {
      $client_test_result = $this->apiClientHandler->testConnection($this->entity);
      $status = match ($client_test_result->status) {
        ApiClientStatusType::OK => $this->t('OK'),
        ApiClientStatusType::ERROR => $this->t('Error'),
        ApiClientStatusType::INITIALIZING => $this->t('Initializing'),
      };

      $form['status'] = [
        '#type' => 'item',
        '#title' => $this->t('Status'),
        '#markup' => $status,
      ];

      if ($client_test_result->status === ApiClientStatusType::ERROR) {
        $form['status_message'] = [
          '#type' => 'item',
          '#title' => $this->t('Error message'),
          '#markup' => $client_test_result->message,
        ];
      }
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#description' => $this->t('A label to identify the API Client (authentication credentials).'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => [$this, 'loadApiClient'],
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Client ID'),
      '#description' => $this->t('The Client ID of the Brightcove API Authentication credentials, available <a href=":link" target="_blank">here</a>.', [':link' => 'https://studio.brightcove.com/products/videocloud/admin/oauthsettings']),
      '#maxlength' => 255,
      '#default_value' => $this->entity->getClientId(),
      '#required' => TRUE,
    ];

    $form['secret_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Secret Key'),
      '#description' => $this->t('The Secret Key associated with the Client ID above, only visible once when Registering New Application.'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->getSecretKey(),
      '#required' => TRUE,
    ];

    $form['account_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Account ID'),
      '#description' => $this->t('The number of one of the Brightcove accounts "selected for authorization" with the API Client above.'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->getAccountId(),
      '#required' => TRUE,
    ];

    $form['default_player'] = [
      '#type' => 'select',
      '#title' => $this->t('Default player'),
      '#options' => $this->entityTypeManager->getStorage('brightcove_player')->getlist([$this->entity->id()]),
      '#default_value' => $this->entity->getDefaultPlayer() ?: ApiClientInterface::DEFAULT_PLAYER,
      '#required' => TRUE,
    ];
    if ($this->entity->isNew()) {
      $form['default_player']['#description'] = $this->t('The rest of the players will be available after saving.');
    }

    // Count BrightcoveAPIClients.
    $api_clients_number = $this->entityTypeManager->getStorage('brightcove_api_client')->getQuery()->accessCheck()->count()->execute();
    $form['default_client'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Default API Client'),
      '#description' => $this->t('Enable to make this the default API Client.'),
      '#default_value' => $api_clients_number === 0 || ($this->settings->getDefaultApiClientId() === $this->entity->id()),
      '#disabled' => $api_clients_number < 2,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    try {
      // Get API handler by creating a minimal API client config entity.
      $cms_api = $this->cmsApiHandler->getApi($this->getMinimalApiClient($form_state));

      // Test connection by getting a video count.
      $cms_api->countVideos();

      // @todo MOVE this.
      // $this->keyValueExpirableStore->setWithExpire($form_state->getValue('id'), $client->getAccessToken(), intval($client->getExpiresIn()) - 30);
    }
    catch (AuthenticationException $e) {
      $form_state->setErrorByName('client_id', $e->getMessage() ?: $this->t('Invalid credentials.'));
      $form_state->setErrorByName('secret_key');
    }
    catch (ApiException | ClientExceptionInterface $e) {
      $form_state->setErrorByName('account_id', $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $cms_api = $this->cmsApiHandler->getApi($this->getMinimalApiClient($form_state));

    $video_fields = $cms_api->getVideoFields();
    $this->entity->setMaxCustomFields($video_fields->getMaxCustomFields());

    foreach ($video_fields->getCustomFields() as $custom_field) {
      // Create queue item.
      $this->customFieldQueue->createItem([
        'api_client_id' => $this->entity->id(),
        'custom_field' => $custom_field,
      ]);
    }

    parent::submitForm($form, $form_state);

    if ($this->entity->isNew()) {
      // Get Players the first time when the API client is being saved.
      $pm_api = $this->pmApiHandler->getApi($this->getMinimalApiClient($form_state));
      $player_list = $pm_api->listPlayers();
      foreach ($player_list->getItems() as $player) {
        // Create queue item.
        $this->playerQueue->createItem([
          'api_client_id' => $this->entity->id(),
          'player' => $player,
        ]);
      }

      // Get Custom fields the first time when the API client is being saved.
      $video_fields = $cms_api->getVideoFields();
      foreach ($video_fields->getCustomFields() as $custom_field) {
        // Create queue item.
        $this->customFieldQueue->createItem([
          'api_client_id' => $this->entity->id(),
          'custom_field' => $custom_field,
        ]);
      }

      // Create new default subscription.
      $subscription = new Subscription(TRUE);
      $subscription->setStatus(FALSE)
        ->setApiClient($this->entity)
        ->setEndpoint(BrightcoveUtil::getDefaultSubscriptionUrl())
        ->setEvents(['video-change'])
        ->save();

      // Get Subscriptions the first time when the API client is being saved.
      $this->subscriptionsQueue->createItem($this->entity->id());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $status = $this->entityTypeManager->getStorage('brightcove_api_client')->save($this->entity);

    switch ($status) {
      case SAVED_NEW:
        $this->messenger->addStatus($this->t('Created the %label Brightcove API Client.', [
          '%label' => $this->entity->label(),
        ]));

        // Initialize batch.
        batch_set([
          'title' => $this->t('Brightcove sync'),
          'operations' => [
            [
              [$this->queueHandler, 'runQueue'], ['brightcove_player_queue_worker'],
            ],
            [
              [$this->queueHandler, 'runQueue'], ['brightcove_custom_field_queue_worker'],
            ],
            [
              [$this->queueHandler, 'runQueue'], ['brightcove_subscriptions_queue_worker'],
            ],
            [
              [$this->queueHandler, 'runQueue'], ['brightcove_subscription_queue_worker'],
            ],
          ],
        ]);
        break;

      case SAVED_UPDATED:
        $this->messenger->addStatus($this->t('Saved the %label Brightcove API Client.', [
          '%label' => $this->entity->label(),
        ]));
        break;

      default:
        $this->messenger->addError($this->t('There was an error saving the Brightcove API Client.'));
        $form_state->setRebuild(TRUE);
        return;
    }

    if ($form_state->getValue('default_client')) {
      $this->settings->setDefaultApiClientId($this->entity->id());
    }

    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
  }

  /**
   * Callback for machine name exists.
   *
   * @param string $id
   *   API client ID.
   *
   * @return \Drupal\brightcove\Entity\ApiClientInterface|null
   *   API client or NULL if not found.
   */
  public function loadApiClient(string $id): ?ApiClientInterface {
    /** @var \Drupal\brightcove\Entity\ApiClientInterface|null $api_client */
    $api_client = $this->entityTypeManager->getStorage('brightcove_api_client')->load($id);
    return $api_client;
  }

  /**
   * Gets a minimally initializes API client for the API handler.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return \Drupal\brightcove\Entity\ApiClientInterface
   *   Minimally initialized API client.
   */
  private function getMinimalApiClient(FormStateInterface $form_state): ApiClientInterface {
    static $api_client = NULL;

    if ($api_client === NULL) {
      /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
      $api_client = $this->entityTypeManager->getStorage('brightcove_api_client')->create([
        'account_id' => $form_state->getValue('account_id'),
        'client_id' => $form_state->getValue('client_id'),
        'secret_key' => $form_state->getValue('secret_key'),
      ]);
    }

    return $api_client;
  }

}
