<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form builder for settings.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Initializes a settings form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   String translation.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    TypedConfigManagerInterface $typedConfigManager,
    MessengerInterface $messenger,
    TranslationInterface $stringTranslation,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configFactory, $typedConfigManager);

    $this->messenger = $messenger;
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('messenger'),
      $container->get('string_translation'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'brightcove.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'brightcove_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $api_client_storage = $this->entityTypeManager->getStorage('brightcove_api_client');

    $api_client_ids = $api_client_storage->getQuery()
      ->accessCheck()
      ->execute();

    $api_clients = [];
    foreach ($api_client_storage->loadMultiple($api_client_ids) as $api_client) {
      $api_clients[$api_client->id()] = $api_client->label();
    }

    $form['default_api_client'] = [
      '#type' => 'select',
      '#title' => 'Default API client',
      '#options' => $api_clients,
      '#description' => $this->t('The API client that should be used by default unless told otherwise.'),
      '#config_target' => 'brightcove.settings:defaultAPIClient',
    ];

    if ($api_clients === []) {
      $form['default_api_client']['#empty_value'] = '';
      $form['default_api_client']['#empty_option'] = $this->t('No API clients created yet');
    }

    $form['notification_callback_expiration'] = [
      '#type' => 'number',
      '#title' => $this->t('Notification callback expiration time'),
      '#description' => $this->t('Expiration time in seconds for a notification callback.'),
      '#config_target' => 'brightcove.settings:notification.callbackExpirationTime',
    ];

    $form['marked_field_expiry'] = [
      '#type' => 'number',
      '#title' => $this->t('Marked field expiration'),
      '#description' => $this->t('Expiry time for marked fields in seconds.'),
      '#config_target' => 'brightcove.settings:ingestion.marked_field_expiry',
    ];

    $form['disable_cron'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable cron'),
      '#description' => $this->t('Enabling this option will prevent a Brightcove-to-Drupal sync running from cron.'),
      '#config_target' => 'brightcove.settings:disable_cron',
    ];

    return parent::buildForm($form, $form_state);
  }

}
