<?php

declare(strict_types=1);

namespace Drupal\brightcove\Form;

use Drupal\brightcove\Entity\Subscription;
use Drupal\brightcove\Services\QueueHandlerInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Queue\QueueFactory;

/**
 * Form builder for the status overview.
 */
class StatusOverviewForm extends FormBase {

  use DependencySerializationTrait;

  /**
   * Initializes a status overview form.
   *
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   The queue factory.
   * @param \Drupal\brightcove\Services\QueueHandlerInterface $queueHandler
   *   Queue handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   String translation.
   */
  public function __construct(
    protected readonly QueueFactory $queueFactory,
    protected readonly QueueHandlerInterface $queueHandler,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('queue'),
      $container->get('brightcove.queue_handler'),
      $container->get('entity_type.manager'),
      $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'queue_overview_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $video_num = $this->entityTypeManager->getStorage('brightcove_video')->getQuery()->accessCheck()->count()->execute();
    $playlist_num = $this->entityTypeManager->getStorage('brightcove_playlist')->getQuery()->accessCheck()->count()->execute();
    $subscription_num = Subscription::count();

    $counts = [
      'client' => $this->entityTypeManager->getStorage('brightcove_api_client')->getQuery()->accessCheck()->count()->execute(),
      'subscription' => $subscription_num,
      'subscription_delete' => $subscription_num,
      'video' => $video_num,
      'video_delete' => $video_num,
      'text_track' => $this->entityTypeManager->getStorage('brightcove_text_track')->getQuery()->accessCheck()->count()->execute(),
      'playlist' => $playlist_num,
      'playlist_delete' => $playlist_num,
      'player' => $this->entityTypeManager->getStorage('brightcove_player')->getQuery()->accessCheck()->count()->execute(),
      'custom_field' => $this->entityTypeManager->getStorage('brightcove_custom_field')->getQuery()->accessCheck()->count()->execute(),
    ];

    $queues = [
      'client' => $this->t('Client'),
      'subscription' => $this->t('Subscription'),
      'player' => $this->t('Player'),
      'custom_field' => $this->t('Custom field'),
      'video' => $this->t('Video'),
      'text_track' => $this->t('Text Track'),
      'playlist' => $this->t('Playlist'),
      'video_delete' => $this->t('Check deleted videos *'),
      'playlist_delete' => $this->t('Check deleted playlists *'),
      'subscription_delete' => $this->t('Check deleted subscriptions'),
    ];

    // There is no form element (i.e. widget) in the table, so it's safe to
    // return a render array for a table as a part of the form build array.
    $form['queues'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Name'),
        $this->t('Number of entities'),
        $this->t('Item(s) in queue'),
      ],
      '#rows' => [],
    ];
    foreach ($queues as $queue => $title) {
      $form['queues']['#rows'][$queue] = [
        $title,
        $counts[$queue],
        $this->queueFactory->get("brightcove_{$queue}_queue_worker")
          ->numberOfItems(),
      ];
    }

    $form['notice'] = [
      '#type' => 'item',
      '#markup' => '<em>* ' . $this->t('May run slowly with lots of items.') . '</em>',
    ];

    $form['sync'] = [
      '#name' => 'sync',
      '#type' => 'submit',
      '#value' => $this->t('Sync all'),
    ];
    $form['run'] = [
      '#name' => 'run',
      '#type' => 'submit',
      '#value' => $this->t('Run all queues'),
    ];
    $form['clear'] = [
      '#name' => 'clear',
      '#type' => 'submit',
      '#value' => $this->t('Clear all queues'),
      '#description' => $this->t('Remove all items from all queues'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    if ($triggering_element = $form_state->getTriggeringElement()) {
      $this->queueHandler->runStatusQueues($triggering_element['#name']);
    }
  }

}
