<?php

declare(strict_types = 1);

namespace Drupal\brightcove;

use Brightcove\Storage\StorageInterface;
use Brightcove\Type\AccessToken;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface;

/**
 * Storage implementation for brightcove/api.
 */
final readonly class LibraryStorage implements StorageInterface {

  /**
   * Initializes a library storage.
   *
   * @param \Drupal\brightcove\Entity\ApiClientInterface $apiClient
   *   API client.
   * @param \Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface $expirableKeyValueStorage
   *   Modified expirable key-value storage.
   */
  public function __construct(
    private ApiClientInterface $apiClient,
    private KeyValueStoreExpirableInterface $expirableKeyValueStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getAccessToken(): ?AccessToken {
    return $this->expirableKeyValueStorage->get("brightcove_access_token.{$this->apiClient->id()}");
  }

  /**
   * {@inheritdoc}
   */
  public function isAccessTokenExpired(): bool {
    return !$this->expirableKeyValueStorage->has("brightcove_access_token.{$this->apiClient->id()}");
  }

  /**
   * {@inheritdoc}
   */
  public function storeAccessToken(AccessToken $accessToken): void {
    $this->expirableKeyValueStorage->setWithExpire("brightcove_access_token.{$this->apiClient->id()}", $accessToken, $accessToken->expiresIn);
  }

}
