<?php

declare(strict_types=1);

namespace Drupal\brightcove;

use Drupal\Core\Entity\EntityInterface;

/**
 * Provides an interface for defining Brightcove Playlists.
 */
interface PlaylistInterface extends EntityInterface, CmsEntityInterface {

  /**
   * Manual playlist type.
   */
  public const int TYPE_MANUAL = 0;

  /**
   * Smart playlist type.
   */
  public const int TYPE_SMART = 1;

  /**
   * Tag search condition "one_or_more".
   */
  public const string TAG_SEARCH_CONTAINS_ONE_OR_MORE = 'contains_one_or_more';

  /**
   * Tag search condition "all".
   */
  public const string TAG_SEARCH_CONTAINS_ALL = 'contains_all';

  /**
   * Returns the playlist type.
   *
   * @return string
   *   The type of the playlist.
   */
  public function getType(): string;

  /**
   * Sets the playlist's type.
   *
   * @param string $type
   *   The type of the playlist, either BrightcovePlaylist::TYPE_MANUAL or
   *   BrightcovePlaylist::SMART.
   *
   * @return \Drupal\brightcove\PlaylistInterface
   *   The called Brightcove Playlist.
   *
   * @throws \InvalidArgumentException
   *   If the value input is inappropriate.
   */
  public function setType(string $type): PlaylistInterface;

  /**
   * Returns the Brightcove Playlist favorite indicator.
   *
   * Favorite Brightcove Playlists are displayed in the sidebar.
   *
   * @return bool
   *   TRUE if the Brightcove Playlist is favorite.
   */
  public function isFavorite(): bool;

  /**
   * Returns the tags search condition for smart playlist.
   *
   * @return string
   *   The condition of the tag search.
   */
  public function getTagsSearchCondition(): string;

  /**
   * Sets the tags search condition for smart playlist.
   *
   * @param string $condition
   *   The condition of the tag search, possible values are:
   *     - TAG_SEARCH_CONTAINS_ONE_OR_MORE
   *     - TAG_SEARCH_CONTAINS_ALL
   *   .
   *
   * @return \Drupal\brightcove\PlaylistInterface
   *   The called Brightcove Playlist.
   */
  public function setTagsSearchCondition(string $condition): PlaylistInterface;

  /**
   * Returns the list of videos on the playlist.
   *
   * @return int[]
   *   The videos on the playlist.
   */
  public function getVideos(): array;

  /**
   * Sets the playlist's videos.
   *
   * @param array|null $videos
   *   The videos on the playlist. Array of ['target_id' => int] items.
   *
   * @return \Drupal\brightcove\PlaylistInterface
   *   The called Brightcove Playlist.
   */
  public function setVideos(?array $videos): PlaylistInterface;

}
