<?php

declare(strict_types=1);

namespace Drupal\brightcove;

use Drupal\Core\Link;
use Drupal\Core\Datetime\DateFormatter;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Session\AccountProxy;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a listing of Brightcove Playlists.
 */
final class PlaylistListBuilder extends EntityListBuilder {

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entityType
   *   Entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   Entity Storage.
   * @param \Drupal\Core\Session\AccountProxy $accountProxy
   *   Account proxy.
   * @param \Drupal\Core\Datetime\DateFormatter $dateFormatter
   *   Date formatter.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   String translation.
   */
  public function __construct(
    EntityTypeInterface $entityType,
    EntityStorageInterface $storage,
    private readonly AccountProxy $accountProxy,
    private readonly DateFormatter $dateFormatter,
    TranslationInterface $stringTranslation,
  ) {
    parent::__construct($entityType, $storage);

    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): self {
    return new self(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('current_user'),
      $container->get('date.formatter'),
      $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntityIds(): array {
    $query = $this->getStorage()->getQuery()
      ->accessCheck()
      ->sort('changed', 'DESC');

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $query->pager($this->limit);
    }
    return $query->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    // Assemble header columns.
    $header = [
      'name' => $this->t('Name'),
      'updated' => $this->t('Updated'),
      'reference_id' => $this->t('Reference ID'),
    ];

    // Add operations header column only if the user has access.
    if ($this->accountProxy->hasPermission('edit brightcove playlists') || $this->accountProxy->hasPermission('delete brightcove playlists')) {
      $header += parent::buildHeader();
    }

    return $header;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\brightcove\Entity\Playlist $entity */
    if (($entity->isPublished() && $this->accountProxy->hasPermission('view published brightcove playlists')) || (!$entity->isPublished() && $this->accountProxy->hasPermission('view unpublished brightcove playlists'))) {
      $name = Link::fromTextAndUrl($entity->label(), new Url(
        'entity.brightcove_playlist.canonical', [
          'brightcove_playlist' => $entity->id(),
        ],
      ));
    }
    else {
      $name = $entity->label();
    }

    // Assemble row.
    $row = [
      'name' => $name,
      'updated' => $this->dateFormatter->format($entity->getChangedTime(), 'short'),
      'reference_id' => $entity->getReferenceId(),
    ];

    // Add operations column only if the user has access.
    if ($this->accountProxy->hasPermission('edit brightcove playlists') || $this->accountProxy->hasPermission('delete brightcove playlists')) {
      $row += parent::buildRow($entity);
    }

    return $row;
  }

}
