<?php

declare(strict_types=1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\CustomFieldInterface;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Update Tasks for API client.
 */
#[QueueWorker(
  id: 'brightcove_client_queue_worker',
  title: new TranslatableMarkup('Brightcove API Client queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class ApiClientQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new BrightcoveClientQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Queue\QueueInterface $videoPageQueue
   *   The video page queue object.
   * @param \Drupal\Core\Queue\QueueInterface $playlistPageQueue
   *   The playlist page queue object.
   * @param \Drupal\Core\Queue\QueueInterface $playerQueue
   *   The player queue object.
   * @param \Drupal\Core\Queue\QueueInterface $playerDeleteQueue
   *   The player delete queue object.
   * @param \Drupal\Core\Queue\QueueInterface $customFieldQueue
   *   The custom field queue object.
   * @param \Drupal\Core\Queue\QueueInterface $customFieldDeleteQueue
   *   The custom field queue object.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    protected readonly QueueInterface $videoPageQueue,
    protected readonly QueueInterface $playlistPageQueue,
    protected readonly QueueInterface $playerQueue,
    protected readonly QueueInterface $playerDeleteQueue,
    protected readonly QueueInterface $customFieldQueue,
    protected readonly QueueInterface $customFieldDeleteQueue,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $entity_type_manager = $container->get('entity_type.manager');
    $queue = $container->get('queue');

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $queue->get('brightcove_video_page_queue_worker'),
      $queue->get('brightcove_playlist_page_queue_worker'),
      $queue->get('brightcove_player_queue_worker'),
      $queue->get('brightcove_player_delete_queue_worker'),
      $queue->get('brightcove_custom_field_queue_worker'),
      $queue->get('brightcove_custom_field_delete_queue_worker'),
      $entity_type_manager,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $cms = BrightcoveUtil::getCmsApi($data);
    $items_per_page = 20;

    // Create queue item for each player.
    $pm = BrightcoveUtil::getPmApi($data);
    $player_list = $pm->listPlayers();
    $players = $player_list->getItems();

    $player_entities = $this->entityTypeManager->getStorage('brightcove_player')->getList([$data]);
    foreach ($players as $player) {
      // Remove existing players from the list.
      unset($player_entities[$player->getId()]);

      // Create queue item.
      $this->playerQueue->createItem([
        'api_client_id' => $data,
        'player' => $player,
      ]);
    }
    // Remove non-existing players.
    foreach (array_keys($player_entities) as $player_id) {
      // Create queue item for deletion.
      $this->playerDeleteQueue->createItem(['player_id' => $player_id]);
    }

    // Create queue item for each custom field.
    $video_fields = $cms->getVideoFields();
    $custom_fields = [];
    foreach ($video_fields->getCustomFields() as $custom_field) {
      $custom_fields[] = $custom_field->getId();
      // Create queue item.
      $this->customFieldQueue->createItem([
        'api_client_id' => $data,
        'custom_field' => $custom_field,
      ]);
    }
    // Collect non-existing custom fields and delete them.
    /** @var \Drupal\brightcove\CustomFieldInterface[] $custom_field_entities */
    $custom_field_entities = $this->entityTypeManager->getStorage('brightcove_custom_field')->loadByProperties([
      'api_client' => $data,
      'status' => CustomFieldInterface::STATUS_ACTIVE,
    ]);
    foreach ($custom_field_entities as $custom_field_entity) {
      if (!in_array($custom_field_entity->getCustomFieldId(), $custom_fields)) {
        $this->customFieldDeleteQueue->createItem($custom_field_entity);
      }
    }

    // Create queue items for each video page.
    $video_count = $cms->countVideos();
    $page = 0;
    while ($page * $items_per_page < $video_count) {
      $this->videoPageQueue->createItem([
        'api_client_id' => $data,
        'page' => $page,
        'items_per_page' => $items_per_page,
      ]);
      $page++;
    }

    // Create queue items for each playlist page.
    $playlist_count = $cms->countPlaylists();
    $page = 0;
    while ($page * $items_per_page < $playlist_count) {
      $this->playlistPageQueue->createItem([
        'api_client_id' => $data,
        'page' => $page,
        'items_per_page' => $items_per_page,
      ]);
      $page++;
    }
  }

}
