<?php

declare(strict_types=1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Entity\CustomField;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Delete Tasks for Custom Fields.
 */
#[QueueWorker(
  id: 'brightcove_custom_field_delete_queue_worker',
  title: new TranslatableMarkup('Brightcove Custom Field delete queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class CustomFieldDeleteQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Initializes a custom field delete queue worker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $custom_field_storage = $this->entityTypeManager->getStorage('brightcove_custom_field');

    // Delete custom field.
    if ($data instanceof CustomField) {
      $custom_field_storage->delete([$data]);
    }
    else {
      /** @var \Drupal\brightcove\Entity\CustomField $custom_field_entity */
      $custom_field_entity = $custom_field_storage->load($data);

      if (!is_null($custom_field_entity)) {
        $custom_field_storage->delete([$custom_field_entity]);
      }
    }
  }

}
