<?php

declare(strict_types=1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Delete Tasks for Player.
 */
#[QueueWorker(
  id: 'brightcove_player_delete_queue_worker',
  title: new TranslatableMarkup('Brightcove Player delete queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class PlayerDeleteQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new PlayerDeleteQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $player_storage = $this->entityTypeManager->getStorage('brightcove_player');

    if (isset($data['player_id'])) {
      /** @var \Drupal\brightcove\PlayerInterface[] $players */
      $players = $player_storage->loadByProperties([
        'player_id' => $data['player_id'],
      ]);
      $player = reset($players);

      if (!is_null($player)) {
        $player_storage->delete([$player]);
      }
    }
    elseif (isset($data['player_entity_id'])) {
      /** @var \Drupal\brightcove\Entity\Player $player */
      $player = $player_storage->load($data['player_entity_id']);

      if (!is_null($player)) {
        $player_storage->delete([$player]);
      }
    }
  }

}
