<?php

declare(strict_types=1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Entity\Player;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Update Tasks for Players.
 */
#[QueueWorker(
  id: 'brightcove_player_queue_worker',
  title: new TranslatableMarkup('Brightcove Player queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class PlayerQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new BrightcovePlayerQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    /** @var \Brightcove\Item\Player\Player $player */
    $player = $data['player'];

    /** @var \Drupal\brightcove\Entity\Storage\PlayerStorageInterface $player_storage */
    $player_storage = $this->entityTypeManager->getStorage('brightcove_player');
    Player::createOrUpdate($player, $player_storage, $data['api_client_id']);
  }

}
