<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Services\CmsApiHandlerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Delete Tasks for Playlist.
 */
#[QueueWorker(
  id: 'brightcove_playlist_delete_queue_worker',
  title: new TranslatableMarkup('Brightcove Playlist delete queue worker'),
  cron: [
    'time' => 30,
  ]
)]
class PlaylistDeleteQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Initializes a Playlist delete queue worker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\brightcove\Services\CmsApiHandlerInterface $cmsApiHandler
   *   CMS API handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    private readonly CmsApiHandlerInterface $cmsApiHandler,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('brightcove.api.cms'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    [$api_client_id, , $playlists] = $data;

    // Check the playlist if it is available on Brightcove or not.
    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = $this->entityTypeManager->getStorage('brightcove_api_client')->load($api_client_id);
    $cms_api = $this->cmsApiHandler->getApi($api_client);

    $playlist_ids = [];
    foreach ($playlists as $playlist) {
      $playlist_ids[] = $playlist->playlist_id;
    }

    $remote_playlists = $cms_api->getPlaylistsByIds($playlist_ids);

    // Check for missing videos if any is missing from the remote.
    if (count($playlists) !== count($remote_playlists)) {
      $missing_playlists = [];
      foreach ($playlists as $playlist) {
        foreach ($remote_playlists as $remote_playlist) {
          if ($remote_playlist->getId() === $playlist->playlist_id) {
            continue 2;
          }
        }

        $missing_playlists[] = $playlist->bcplid;
      }

      // Delete missing videos locally.
      if ($missing_playlists !== []) {
        $playlist_storage = $this->entityTypeManager->getStorage('brightcove_playlist');
        $playlist_storage->delete($playlist_storage->loadMultiple($missing_playlists), FALSE);
      }
    }
  }

}
