<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Services\CmsApiHandlerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Update Tasks for Playlist.
 */
#[QueueWorker(
  id: 'brightcove_playlist_page_queue_worker',
  title: new TranslatableMarkup('Brightcove Playlist page queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class PlaylistPageQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new BrightcovePlaylistPageQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   Queue factory.
   * @param \Drupal\brightcove\Services\CmsApiHandlerInterface $cmsApiHandler
   *   CMS API handler.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly QueueFactory $queueFactory,
    private readonly CmsApiHandlerInterface $cmsApiHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('queue'),
      $container->get('brightcove.api.cms'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = $this->entityTypeManager->getStorage('brightcove_api_client')->load($data['api_client_id']);
    $cms_api = $this->cmsApiHandler->getApi($api_client);

    // Get playlists.
    $playlists = $cms_api->listPlaylists(NULL, $data['items_per_page'], $data['page'] * $data['items_per_page']);
    foreach ($playlists as $playlist) {
      $this->queueFactory->get('brightcove_playlist_queue_worker')->createItem([
        'api_client_id' => $data['api_client_id'],
        'playlist' => $playlist,
      ]);
    }
  }

}
