<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Services\PlaylistHelperInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Queue\SuspendQueueException;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Update Tasks for Playlist.
 */
#[QueueWorker(
  id: 'brightcove_playlist_queue_worker',
  title: new TranslatableMarkup('Brightcove Playlist queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class PlaylistQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new BrightcovePlaylistQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\brightcove\Services\PlaylistHelperInterface $playlistHelper
   *   Playlist helper.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    private readonly PlaylistHelperInterface $playlistHelper,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('brightcove.playlist.helper'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    /** @var \Brightcove\Item\Playlist $brightcove_playlist */
    $brightcove_playlist = $data['playlist'];

    try {
      $this->playlistHelper->createOrUpdate($brightcove_playlist, $data['api_client_id']);
    }
    catch (\Exception $e) {
      throw new SuspendQueueException($e->getMessage(), $e->getCode(), $e);
    }
  }

}
