<?php

declare(strict_types=1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Brightcove\API\Exception\ApiException;
use Brightcove\Item\Subscription;
use Drupal\brightcove\BrightcoveUtil;
use Drupal\brightcove\Services\LoggerInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Gathers the subscriptions for creation and delete check.
 */
#[QueueWorker(
  id: 'brightcove_subscriptions_queue_worker',
  title: new TranslatableMarkup('Brightcove Subscriptions queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class SubscriptionsQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new BrightcoveSubscriptionQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Queue\QueueInterface $subscriptionQueue
   *   The brightcove_subscription queue.
   * @param \Drupal\brightcove\Services\LoggerInterface $logger
   *   Logger.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    protected readonly QueueInterface $subscriptionQueue,
    protected readonly LoggerInterface $logger,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('queue')->get('brightcove_subscription_queue_worker'),
      $container->get('brightcove.logger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $cms = BrightcoveUtil::getCmsApi($data);
    try {
      $subscriptions = $cms->getSubscriptions();

      foreach (!empty($subscriptions) ? $subscriptions : [] as $subscription) {
        if ($subscription instanceof Subscription) {
          $this->subscriptionQueue->createItem([
            'api_client_id' => $data,
            'subscription' => $subscription,
          ]);
        }
      }
    }
    catch (ApiException $e) {
      if ($e->getCode() === 401) {
        $this->logger->logException($e, 'Access denied for Notifications.', [], RfcLogLevel::WARNING);
      }
      else {
        throw $e;
      }
    }
  }

}
