<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Services\CmsApiHandlerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Delete Tasks for Video.
 */
#[QueueWorker(
  id: 'brightcove_video_delete_queue_worker',
  title: new TranslatableMarkup('Brightcove Video delete queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class VideoDeleteQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new BrightcoveVideoQueueWorker object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\brightcove\Services\CmsApiHandlerInterface $cmsApiHandler
   *   CMS API handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    private readonly CmsApiHandlerInterface $cmsApiHandler,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('brightcove.api.cms'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    [$api_client_id, $limit, $videos] = $data;

    /** @var \Drupal\brightcove\Entity\ApiClientInterface $api_client */
    $api_client = $this->entityTypeManager->getStorage('brightcove_api_client')->load($api_client_id);

    // Check the videos if they are available on Brightcove or not.
    $cms_api = $this->cmsApiHandler->getApi($api_client);
    $video_ids = '';
    foreach ($videos as $video) {
      $video_ids .= "id:{$video->video_id} ";
    }
    $remote_videos = $cms_api->listVideos(rtrim($video_ids), limit: $limit);

    // Check for missing videos if any is missing from the remote.
    if (count($videos) !== count($remote_videos)) {
      $missing_videos = [];
      foreach ($videos as $video) {
        foreach ($remote_videos as $remote_video) {
          if ($remote_video->getId() === $video->video_id) {
            continue 2;
          }
        }

        $missing_videos[] = $video->bcvid;
      }

      // Delete missing videos locally.
      if ($missing_videos !== []) {
        $video_storage = $this->entityTypeManager->getStorage('brightcove_video');
        $video_storage->delete($video_storage->loadMultiple($missing_videos), FALSE);
      }
    }
  }

}
