<?php

declare(strict_types=1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\brightcove\BrightcoveUtil;

/**
 * Processes Entity Update Tasks for Video.
 */
#[QueueWorker(
  id: 'brightcove_video_page_queue_worker',
  title: new TranslatableMarkup('Brightcove Video page queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class VideoPageQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Initializes a video page queue worker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   Queue factory.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    private readonly QueueFactory $queueFactory,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('queue'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $cms = BrightcoveUtil::getCmsApi($data['api_client_id']);

    // Get videos.
    $videos = $cms->listVideos(NULL, 'created_at', $data['items_per_page'], $data['page'] * $data['items_per_page']);

    // Put acquired videos into the queue.
    foreach ($videos as $video) {
      $this->queueFactory->get('brightcove_video_queue_worker')->createItem([
        'api_client_id' => $data['api_client_id'],
        'video' => $video,
      ]);
    }
  }

}
