<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Plugin\QueueWorker;

use Drupal\brightcove\Services\VideoHelperInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\Attribute\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes Entity Update Tasks for Video.
 */
#[QueueWorker(
  id: 'brightcove_video_queue_worker',
  title: new TranslatableMarkup('Brightcove Video queue worker'),
  cron: [
    'time' => 30,
  ],
)]
class VideoQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Initializes a video queue worker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $pluginId
   *   The plugin_id for the plugin instance.
   * @param mixed $pluginDefinition
   *   The plugin implementation definition.
   * @param \Drupal\brightcove\Services\VideoHelperInterface $videoHelper
   *   Video helper.
   */
  public function __construct(
    array $configuration,
    string $pluginId,
    mixed $pluginDefinition,
    private readonly VideoHelperInterface $videoHelper,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('brightcove.video_helper'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    /** @var \Brightcove\Item\Video\Video $brightcove_video */
    $brightcove_video = $data['video'];

    $video = $this->videoHelper->createOrUpdate($brightcove_video, $data['api_client_id']);
    $this->videoHelper->queueTextTracks($video, $brightcove_video);
  }

}
