<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Services;

use Brightcove\API\ApiClient;
use Brightcove\API\ApiClientInterface as BrightcoveApiClientInterface;
use Brightcove\API\CMS;
use Brightcove\API\Exception\ApiException;
use Drupal\brightcove\Entity\ApiClientInterface;
use Drupal\brightcove\Event\HttpClientConfigEvent;
use Drupal\brightcove\Type\ApiClientStatus;
use Drupal\brightcove\Type\ApiClientStatusType;
use Drupal\Core\Http\ClientFactory;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * API client handler service.
 */
final class ApiClientHandler implements ApiClientHandlerInterface {

  /**
   * Authorized API clients.
   *
   * @var \Brightcove\API\ApiClientInterface[]
   */
  private static array $clients = [];

  /**
   * Initializes an API client handler.
   *
   * @param \Drupal\Core\Http\ClientFactory $clientFactory
   *   HTTP client factory.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   Event dispatcher.
   * @param \Drupal\brightcove\Services\LibraryStorageFactoryInterface $libraryStorageFactory
   *   Library storage factory.
   * @param \Drupal\brightcove\Services\LoggerInterface $logger
   *   Logger.
   */
  public function __construct(
    private readonly ClientFactory $clientFactory,
    private readonly EventDispatcherInterface $eventDispatcher,
    private readonly LibraryStorageFactoryInterface $libraryStorageFactory,
    private readonly LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function getClient(ApiClientInterface $api_client, bool $reconnect = FALSE): BrightcoveApiClientInterface {
    $api_client_id = $api_client->id();
    if ($reconnect || isset(self::$clients[$api_client_id])) {
      return self::$clients[$api_client_id];
    }

    $config = new HttpClientConfigEvent();
    $this->eventDispatcher->dispatch($config);

    return self::$clients[$api_client_id] = new ApiClient($this->clientFactory->fromOptions($config->getAll()), $api_client->getClientId(), $api_client->getSecretKey(), $this->libraryStorageFactory->get($api_client));
  }

  /**
   * {@inheritdoc}
   */
  public function testConnection(ApiClientInterface $api_client, bool $reconnect = FALSE): ApiClientStatus {
    try {
      // Test API connection by calling the count videos endpoint.
      $cms = new CMS($this->getClient($api_client, $reconnect), $api_client->getAccountId());
      $cms->countVideos();
    }
    catch (\Exception $e) {
      if ($e instanceof ApiException) {
        // If we got an unauthorized error, try to re-authorize the client
        // only once.
        if ($e->getCode() === 401 && !$reconnect) {
          return $this->testConnection($api_client, TRUE);
        }
      }

      // If we have already tried to re-authenticate the client, log the error.
      $this->logger->logException($e, 'Failed to authorize client with Brightcove API.');

      // Return with an error status.
      return new ApiClientStatus(ApiClientStatusType::ERROR, 'Failed to authorize client with Brightcove API. See logs for additional details.');
    }

    return new ApiClientStatus(ApiClientStatusType::OK);
  }

}
