<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Services;

use Brightcove\Item\Playlist;
use Drupal\brightcove\PlaylistInterface;

/**
 * Helper service for Playlist entity.
 */
final readonly class PlaylistHelper extends VideoPlaylistHelperBase implements PlaylistHelperInterface {

  /**
   * {@inheritdoc}
   */
  public function createOrUpdate(Playlist $brightcove_playlist, string $api_client_id): void {
    // May throw an \Exception if any of the videos is not found.
    $videos = $this->extractVideosArray($brightcove_playlist, $api_client_id);

    // Only save the brightcove_playlist entity if it's really updated or
    // created now.
    $needs_save = FALSE;

    $playlist_storage = $this->entityTypeManager->getStorage('brightcove_playlist');

    // Try to get an existing playlist.
    $existing_playlist = $playlist_storage->getQuery()
      ->condition('api_client', $api_client_id)
      ->condition('playlist_id', $brightcove_playlist->getId())
      ->accessCheck()
      ->execute();

    $updated_at = $brightcove_playlist->getUpdatedAt();
    $updated_at = $updated_at !== NULL ? strtotime($updated_at) : 0;

    // Update existing playlist if needed.
    if (!empty($existing_playlist)) {
      // Load Brightcove Playlist.
      $playlist_entity_id = reset($existing_playlist);
      /** @var \Drupal\brightcove\PlaylistInterface $playlist */
      $playlist = $playlist_storage->load($playlist_entity_id);

      // Update playlist if it is changed on Brightcove.
      if ($playlist->getChangedTime() < $updated_at) {
        $needs_save = TRUE;
        // Update changed time.
        $playlist->setChangedTime($updated_at);
      }
    }
    // Create playlist if it does not exist.
    else {
      $created_at = $brightcove_playlist->getCreatedAt();
      $needs_save = TRUE;
      // Create new Brightcove Playlist entity.
      /** @var \Drupal\brightcove\PlaylistInterface $playlist */
      $playlist = $playlist_storage->create([
        'api_client' => [
          'target_id' => $api_client_id,
        ],
        'playlist_id' => $brightcove_playlist->getId(),
        'created' => $created_at !== NULL ? strtotime($created_at) : 0,
        'changed' => $updated_at,
      ]);
    }

    if ($needs_save) {
      // Update type field if needed.
      if ($playlist->getType() !== ($type = $brightcove_playlist->getType())) {
        $playlist->setType($type);
      }

      // Update name field if needed.
      if ($playlist->getName() !== ($name = $brightcove_playlist->getName())) {
        $playlist->setName($name);
      }

      // Update favorite field if needed.
      if ($playlist->isFavorite() !== ($favorite = $brightcove_playlist->isFavorite())) {
        // This is a non-modifiable field, so it does not have a specific
        // setter.
        $playlist->set('favorite', $favorite);
      }

      // Update reference ID field if needed.
      if ($playlist->getReferenceId() !== ($reference_id = $brightcove_playlist->getReferenceId())) {
        $playlist->setReferenceId($reference_id);
      }

      // Update description field if needed.
      if ($playlist->getDescription() !== ($description = $brightcove_playlist->getDescription())) {
        $playlist->setDescription($description);
      }

      // Update tags field if needed.
      $playlist_search = $brightcove_playlist->getSearch();
      if ($playlist_search !== NULL) {
        preg_match('/^(\+?)([^+].*):(?:,?"(.*?[^"])")$/i', $playlist_search, $matches);
        if (count($matches) === 4 && $matches[2] === 'tags') {
          $playlist_tags = explode('","', $matches[3]);

          // Save or update tag search condition if needed.
          $playlist_search_condition = $matches[1] === '+' ? PlaylistInterface::TAG_SEARCH_CONTAINS_ALL : PlaylistInterface::TAG_SEARCH_CONTAINS_ONE_OR_MORE;
          if ($playlist->getTagsSearchCondition() !== $playlist_search_condition) {
            $playlist->setTagsSearchCondition($playlist_search_condition);
          }

          $this->saveOrUpdateTags($playlist, $api_client_id, $playlist_tags);
        }
      }

      // Update videos field if needed.
      if ($playlist->getVideos() !== $videos) {
        $playlist->setVideos($videos);
      }

      // @todo State/published.
      $playlist_storage->save($playlist);
    }
  }

  /**
   * Converts videos from \Brightcove\Item\Playlist to Drupal Field API array.
   *
   * @param \Brightcove\Item\Playlist $playlist
   *   The playlist whose videos should be extracted.
   * @param string $api_client_id
   *   API client ID.
   *
   * @return array
   *   The Drupal Field API array that can be saved into a multivalue
   *   entity_reference field (like brightcove_playlist.videos).
   *
   * @throws \Exception
   *   Thrown when any of the videos is unavailable on the Drupal side.
   */
  private function extractVideosArray(Playlist $playlist, string $api_client_id): array {
    $brightcove_video_ids = $playlist->getVideoIds();

    // If the playlist does not contain any videos, finish.
    if ($brightcove_video_ids === []) {
      return [];
    }

    // Get videos by their Brightcove ID.
    $video_ids = $this->entityTypeManager->getStorage('brightcove_video')->getQuery()
      ->accessCheck(FALSE)
      ->condition('api_client', $api_client_id)
      ->condition('video_id', $brightcove_video_ids, 'IN')
      ->execute();

    if (count($brightcove_video_ids) !== count($video_ids)) {
      // If a video is not found, then throw an exception that will effectively
      // stop the queue worker, so the playlist with any "unknown" video will
      // remain in the queue, so it could be picked up the next time, hoping the
      // video will be available by then.
      throw new \Exception('Playlist contains a Video that is not (yet) available in Drupal.');
    }

    $return = [];
    foreach ($video_ids as $video_id) {
      $return[] = [
        'target_id' => $video_id,
      ];
    }

    return $return;
  }

}
