<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Services;

use Drupal\brightcove\Exception\BrightcoveUtilException;
use Drupal\brightcove\Services\Exception\QueueHandlerInvalidArgumentException;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueGarbageCollectionInterface;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\Core\Queue\SuspendQueueException;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Http\Client\Exception\HttpException;

/**
 * Handler for queues.
 */
final class QueueHandler implements QueueHandlerInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  public function __construct(
    private readonly LoggerInterface $logger,
    private readonly QueueFactory $queueFactory,
    private readonly QueueWorkerManagerInterface $queueWorkerManager,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function getStatusQueues(): array {
    return [
      'brightcove_client_queue_worker',
      'brightcove_player_queue_worker',
      'brightcove_player_delete_queue_worker',
      'brightcove_custom_field_queue_worker',
      'brightcove_custom_field_delete_queue_worker',
      'brightcove_video_page_queue_worker',
      'brightcove_video_queue_worker',
      'brightcove_text_track_queue_worker',
      'brightcove_text_track_delete_queue_worker',
      'brightcove_playlist_page_queue_worker',
      'brightcove_playlist_queue_worker',
      'brightcove_video_delete_queue_worker',
      'brightcove_playlist_delete_queue_worker',
      'brightcove_subscriptions_queue_worker',
      'brightcove_subscription_queue_worker',
      'brightcove_subscription_delete_queue_worker',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function clearQueue(string $queue_name): void {
    if (!in_array($queue_name, self::getStatusQueues())) {
      throw new QueueHandlerInvalidArgumentException(sprintf('The "%s" queue cannot be cleared by this handler.', $queue_name));
    }

    $this->queueFactory->get($queue_name)->deleteQueue();
  }

  /**
   * {@inheritdoc}
   */
  public function runStatusQueues(string $type): void {
    $queues = self::getStatusQueues();

    $batch_operations = [];

    switch ($type) {
      /* @noinspection PhpMissingBreakStatementInspection */
      case 'sync':
        $batch_operations[] = ['_brightcove_initiate_sync', []];
        // There is intentionally no break here.
      case 'run':
        $callback = [$this, 'runQueue'];
        break;

      case 'clear':
        $callback = [$this, 'clearQueue'];
        break;

      default:
        throw new BrightcoveUtilException('Invalid queue type.');
    }

    // Build queue operations array.
    foreach ($queues as $queue) {
      $batch_operations[] = [$callback, [$queue]];
    }

    if ($batch_operations) {
      // Clean-up expired items in the default queue implementation table. If
      // that's not used, this will simply be a no-op.
      // @see system_cron()
      foreach ($queues as $queue) {
        $queue = $this->queueFactory->get($queue);
        if ($queue instanceof QueueGarbageCollectionInterface) {
          $queue->garbageCollection();
        }
      }

      batch_set([
        'title' => $this->t('Brightcove sync'),
        'operations' => $batch_operations,
      ]);
    }
  }

  /**
   * Runs a queue.
   *
   * @param string $queue
   *   The queue name to clear.
   * @param array &$context
   *   The Batch API context.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function runQueue(string $queue, array &$context): void {
    /** @var \Drupal\Core\Queue\QueueWorkerInterface $queue_worker */
    $queue_worker = $this->queueWorkerManager->createInstance($queue);
    $queue = $this->queueFactory->get($queue);

    if (!isset($context['sandbox']['max'])) {
      $context['sandbox']['max'] = $queue->numberOfItems();
    }

    // Let's process ALL the items in the queue, 5 by 5, to avoid PHP timeouts.
    // If there's any problem with processing any of those 5 items, stop sooner.
    $limit = 5;
    $handled_all = TRUE;
    while (($limit > 0) && ($item = $queue->claimItem(5))) {
      try {
        $queue_worker->processItem($item->data);
        $queue->deleteItem($item);
      }
      catch (SuspendQueueException) {
        $queue->deleteItem($item);
        break;
      }
      catch (HttpException $e) {
        if ($e->getCode() === 401) {
          $queue->deleteItem($item);
          $handled_all = TRUE;
        }
        else {
          $this->logger->logException($e, 'Failed to process queue item on remote object.');
          $handled_all = FALSE;
        }
      }
      catch (\Exception $e) {
        $this->logger->logException($e, 'Failed to process queue item.');
        $handled_all = FALSE;
      }
      $limit--;
    }

    // As this batch may be run synchronously with the queue's cron processor,
    // we can't be sure about the number of items left for the batch as long as
    // there is any. So let's just inform the user about the number of remaining
    // items, as we don't really care if they are processed by this batch
    // processor or the cron one.
    $remaining = $queue->numberOfItems();
    $context['message'] = $this->t('Processing "@title": @count item(s) remaining out of @max.', [
      '@title' => $queue_worker->getPluginDefinition()['title'],
      '@count' => $remaining,
      '@max' => $context['sandbox']['max'],
    ]);
    $context['finished'] = $handled_all && ($remaining === 0);
  }

}
