<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Services;

/**
 * Interface for a queue handler.
 */
interface QueueHandlerInterface {

  /**
   * Gets status queues.
   *
   * These queues are responsible for synchronizing from Brightcove to Drupal
   * (IOW pulling). The order is important.
   *   - The client queue must be run first, that's out of question: this worker
   *     populates most of the other queues.
   *   - Players should be pulled before videos and playlists.
   *   - Custom fields (which means custom field definitions, not values) should
   *     be pulled before videos.
   *   - Text tracks can only be pulled after videos.
   *   - Playlists can only be pulled after videos.
   *   - Custom fields (again: their definitions) have to be deleted before
   *     pulling videos.
   *   - Text tracks have to be deleted before videos are pulled or deleted.
   *
   * @return string[]
   *   A list of status queues.
   */
  public static function getStatusQueues(): array;

  /**
   * Clears a queue.
   *
   * @param string $queue_name
   *   The queue name to clear.
   */
  public function clearQueue(string $queue_name): void;

  /**
   * Runs specific status queues based on the given $type.
   *
   * @param string $type
   *   The queue's type to run, it can be either sync, run or clear.
   *
   * @throws \Drupal\brightcove\Exception\BrightcoveUtilException
   *   Throws an exception if the queue type is invalid.
   */
  public function runStatusQueues(string $type): void;

}
