<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Services;

use Brightcove\Item\Video\Video;
use Drupal\brightcove\Entity\VideoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;

/**
 * Video related helper.
 */
final readonly class VideoHelper extends VideoPlaylistHelperBase implements VideoHelperInterface {

  /**
   * Initializes a video helper.
   *
   * @param \Drupal\brightcove\Services\IngestionInterface $ingestionHelper
   *   Ingestion helper.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   Queue factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    private IngestionInterface $ingestionHelper,
    private QueueFactory $queueFactory,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  public function createOrUpdate(Video $brightcove_video, ?string $api_client_id = NULL): VideoInterface {
    $video_storage = $this->entityTypeManager->getStorage('brightcove_video');

    // Try to get an existing video.
    $existing_video = $video_storage->getQuery()
      ->condition('video_id', $brightcove_video->getId())
      ->accessCheck()
      ->execute();

    $needs_save = FALSE;

    // Update existing video.
    if (!empty($existing_video)) {
      // Load Brightcove Video.
      /** @var \Drupal\brightcove\Entity\VideoInterface $video */
      $video = $video_storage->load(reset($existing_video));

      // Update video if it is changed on Brightcove.
      if ($video->getChangedTime() < strtotime($brightcove_video->getUpdatedAt())) {
        $needs_save = TRUE;
      }
    }
    // Create video if it does not exist.
    else {
      // Make sure we got an api client id when a new video is being created.
      if (is_null($api_client_id)) {
        throw new \Exception('To create a new Video entity, the api_client_id must be given.');
      }

      // Create new Brightcove video entity.
      $values = [
        'video_id' => $brightcove_video->getId(),
        'api_client' => [
          'target_id' => $api_client_id,
        ],
        'created' => strtotime($brightcove_video->getCreatedAt()),
      ];
      /** @var \Drupal\brightcove\Entity\VideoInterface $video */
      $video = $video_storage->create($values);
      $needs_save = TRUE;
    }

    // Save entity only if it is being created or updated.
    if ($needs_save) {
      // Save or update changed time.
      $video->setChangedTime(strtotime($brightcove_video->getUpdatedAt()));

      // Save or update Description field if needed.
      if ($video->getDescription() !== ($description = $brightcove_video->getDescription())) {
        $video->setDescription($description ?? '');
      }

      // Save or update duration field if needed.
      if ($video->getDuration() !== ($duration = $brightcove_video->getDuration())) {
        $video->setDuration($duration);
      }

      // Save or update economics field if needed.
      if ($video->getEconomics() !== ($economics = $brightcove_video->getEconomics())) {
        $video->setEconomics($economics);
      }

      // Save or update tags field if needed.
      $this->saveOrUpdateTags($video, $api_client_id, $brightcove_video->getTags());

      // Get images.
      $images = $brightcove_video->getImages();

      // Save or update thumbnail image if needed.
      if (!$this->ingestionHelper->isFieldMarkedForIngestion($video, VideoInterface::IMAGE_TYPE_THUMBNAIL)) {
        if (!empty($images[VideoInterface::IMAGE_TYPE_THUMBNAIL]) && !empty($images[VideoInterface::IMAGE_TYPE_THUMBNAIL]->getSrc())) {
          $video->saveImage(VideoInterface::IMAGE_TYPE_THUMBNAIL, $images[VideoInterface::IMAGE_TYPE_THUMBNAIL]);
        }
        // Otherwise, leave empty or remove thumbnail from BrightcoveVideo
        // entity.
        else {
          // Delete file.
          $video->setThumbnail(NULL);
        }
      }

      // Save or update poster image if needed.
      if (!$this->ingestionHelper->isFieldMarkedForIngestion($video, VideoInterface::IMAGE_TYPE_POSTER)) {
        if (!empty($images[VideoInterface::IMAGE_TYPE_POSTER]) && !empty($images[VideoInterface::IMAGE_TYPE_POSTER]->getSrc())) {
          $video->saveImage(VideoInterface::IMAGE_TYPE_POSTER, $images[VideoInterface::IMAGE_TYPE_POSTER]);
        }
        // Otherwise, leave empty or remove poster from BrightcoveVideo entity.
        else {
          // Delete file.
          $video->setPoster(NULL);
        }
      }

      // Save or update link url field if needed.
      $link = $brightcove_video->getLink();
      $related_link_field = $video->getRelatedLink() ?: NULL;
      $related_link = [];
      if (empty($related_link_field) && !empty($link)) {
        $related_link['uri'] = $link->getUrl();
        $related_link['title'] = $link->getText();
      }
      elseif (!empty($related_link_field) && !empty($link)) {
        if ($related_link_field['uri'] !== ($url = $link->getUrl())) {
          $related_link['uri'] = $url;
        }

        if ($related_link_field['title'] !== ($title = $link->getText())) {
          $related_link['title'] = $title;
        }
      }
      else {
        $video->setRelatedLink(NULL);
      }
      if (!empty($related_link)) {
        $video->setRelatedLink($related_link);
      }

      // Save or update long description if needed.
      if ($video->getLongDescription() !== ($long_description = $brightcove_video->getLongDescription())) {
        $video->setLongDescription($long_description ?? '');
      }

      // Save or update Name field if needed.
      if ($video->getName() !== ($name = $brightcove_video->getName())) {
        $video->setName($name);
      }

      // Save or update reference ID field if needed.
      if ($video->getReferenceId() !== ($reference_id = $brightcove_video->getReferenceId())) {
        $video->setReferenceId($reference_id);
      }

      // Save or update custom field values.
      if ($video->getCustomFieldValues() !== $custom_fields = $brightcove_video->getCustomFields()) {
        $video->setCustomFieldValues($custom_fields);
      }

      // Save or update schedule dates if needed.
      $schedule = $brightcove_video->getSchedule();
      if (!is_null($schedule)) {
        if ($video->getScheduleStartsAt() !== ($starts_at = $schedule->getStartsAt())) {
          $video->setScheduleStartsAt(!empty($starts_at) ? $this->convertDate($starts_at) : NULL);
        }
        if ($video->getScheduleEndsAt() !== ($ends_at = $schedule->getEndsAt())) {
          $video->setScheduleEndsAt(!empty($ends_at) ? $this->convertDate($ends_at) : NULL);
        }
      }
      else {
        $video->setScheduleStartsAt(NULL);
        $video->setScheduleEndsAt(NULL);
      }

      // Save or update state.
      // We are settings the video as published only if the state is ACTIVE,
      // otherwise it is set as unpublished.
      $state = $brightcove_video->getState() === VideoInterface::STATE_ACTIVE;
      if ($video->isPublished() !== $state) {
        $video->setPublished($state);
      }

      // Save video entity.
      $video_storage->save($video);
    }

    return $video;
  }

  /**
   * {@inheritdoc}
   */
  public function queueTextTracks(VideoInterface $video, Video $brightcove_video): void {
    $existing_text_tracks = [];

    /** @var \Drupal\brightcove\Entity\TextTrack[] $text_tracks */
    $text_tracks = $this->entityTypeManager->getStorage('brightcove_text_track')->loadMultiple(array_column($video->getTextTracks(), 'target_id'));

    // Collect Brightcove IDs.
    foreach ($text_tracks as $text_track) {
      $existing_text_tracks[$text_track->getTextTrackId()] = TRUE;
    }

    // Save Video text tracks.
    $brightcove_text_tracks = $brightcove_video->getTextTracks();
    foreach ($brightcove_text_tracks as $brightcove_text_track) {
      // Remove existing text tracks from the list which are still existing on
      // Brightcove.
      if (isset($existing_text_tracks[$brightcove_text_track->getId()])) {
        unset($existing_text_tracks[$brightcove_text_track->getId()]);
      }

      // Create new queue item for text track.
      $this->queueFactory->get('brightcove_text_track_queue_worker')->createItem([
        'text_track' => $brightcove_text_track,
        'video_entity_id' => $video->id(),
      ]);
    }

    // Remove existing text tracks that are no longer available on Brightcove.
    foreach (array_keys($existing_text_tracks) as $text_track_id) {
      // Create new delete queue item for text track.
      $this->queueFactory->get('brightcove_text_track_delete_queue_worker')->createItem($text_track_id);
    }
  }

  /**
   * Convert Brightcove date make it digestible by Drupal.
   *
   * @param string $brightcove_date
   *   Brightcove date format.
   *
   * @return string|null
   *   Drupal date format.
   */
  private function convertDate(string $brightcove_date): ?string {
    return preg_replace('/\.\d{3}Z$/i', '', $brightcove_date);
  }

}
