<?php

declare(strict_types = 1);

namespace Drupal\brightcove\Services;

use Drupal\brightcove\Entity\VideoInterface;
use Drupal\brightcove\VideoPlaylistCmsEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Helper base for video and playlist helpers.
 */
abstract readonly class VideoPlaylistHelperBase implements VideoPlaylistHelperBaseInterface {

  /**
   * Initializes a base helper.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function saveOrUpdateTags(VideoPlaylistCmsEntityInterface $entity, string $api_client_id, array $tags = []): void {
    $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');

    $entity_tags = [];
    $video_entity_tags = $entity->getTags();
    foreach ($video_entity_tags as $index => $tag) {
      /** @var \Drupal\taxonomy\TermInterface $term */
      $term = $term_storage->load($tag['target_id']);
      if (!is_null($term)) {
        $entity_tags[$term->id()] = $term->getName();
      }
      // Remove non-existing tag references from the video, if there would
      // be any.
      else {
        unset($video_entity_tags[$index]);
        $entity->setTags($video_entity_tags);
      }
    }
    if (array_values($entity_tags) !== $tags) {
      // Remove deleted tags from the video.
      if (!empty($entity->id())) {
        $tags_to_remove = array_diff($entity_tags, $tags);
        foreach (array_keys($tags_to_remove) as $entity_id) {
          unset($entity_tags[$entity_id]);
        }
      }

      // Add new tags.
      $new_tags = array_diff($tags, $entity_tags);
      $entity_tags = array_keys($entity_tags);
      foreach ($new_tags as $tag) {
        $taxonomy_term = NULL;
        $existing_tags = $term_storage->getQuery()
          ->condition('vid', VideoInterface::TAGS_VID)
          ->condition('name', $tag)
          ->accessCheck()
          ->execute();

        // Create new Taxonomy term item.
        if (empty($existing_tags)) {
          $values = [
            'name' => $tag,
            'vid' => VideoInterface::TAGS_VID,
            'brightcove_api_client' => [
              'target_id' => $api_client_id,
            ],
          ];
          $taxonomy_term = $term_storage->create($values);
          $term_storage->save($taxonomy_term);
        }
        $entity_tags[] = isset($taxonomy_term) ? $taxonomy_term->id() : reset($existing_tags);
      }
      $entity->setTags($entity_tags);
    }
  }

}
