<?php

namespace Drupal\browsersync\Hook;

use Drupal\browsersync\BrowsersyncHelper;
use Drupal\browsersync\Form\SettingsForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Form hook implementations for Browsersync.
 */
class BrowsersyncFormHooks {

  use StringTranslationTrait;

  public function __construct(
    protected BrowsersyncHelper $browsersyncHelper,
  ) {}

  /**
   * Implements hook_form_FORM_ID_alter().
   *
   * Adds the Browsersync configuration options to the theme settings form.
   *
   * @see \Drupal\system\Form\ThemeSettingsForm
   */
  #[Hook('form_system_theme_settings_alter')]
  public function systemThemeSettingsAlter(&$form, FormStateInterface $form_state, $form_id): void {
    // Extract the theme key from the form arguments. If not present, it means
    // that we are altering the global theme settings form.
    $args = $form_state->getBuildInfo()['args'];
    $theme_key = !empty($args[0]) ? $args[0] : NULL;

    $form['browsersync'] = [
      '#type' => 'details',
      '#title' => 'Browsersync settings',
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $form['browsersync']['browsersync_enabled'] = [
      '#title' => 'Enable Browsersync',
      '#type' => 'checkbox',
      '#default_value' => $this->browsersyncHelper->browsersyncGetSetting('enabled', $theme_key),
    ];

    $form['browsersync']['settings'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          'input[name="browsersync_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['browsersync']['settings']['browsersync_host'] = [
      '#title' => 'Host',
      '#type' => 'textfield',
      '#description' => $this->t('Override host detection if you know the correct IP to use.'),
      '#default_value' => $this->browsersyncHelper->browsersyncGetSetting('host', $theme_key),
    ];

    $form['browsersync']['settings']['browsersync_port'] = [
      '#title' => 'Port',
      '#type' => 'textfield',
      '#description' => $this->t('Use a specific port (instead of the one auto-detected by Browsersync).'),
      '#default_value' => $this->browsersyncHelper->browsersyncGetSetting('port', $theme_key),
    ];

    $form['#submit'][] = [SettingsForm::class, 'submitForm'];
  }

}
